# /hades/testsuite/simobjects/testSimObject.py
#
# script and functions to check individual SimObjects in Hades.
# The script takes a Java class name as argument, tries to instantiate
# a SimObject of the given type, tries to build the SimObject Symbol,
# checks the SimObject ports, tries to connect/disconnect Signals
# to all ports, tries to copy the SimObject, copies the copy,
# deletes the copies, and finally tries configure() 
#
#
# 23.08.01 fnh
#
from java.lang       import Class
from java.lang       import Thread
from java.awt	       import Point
from java.io         import PrintWriter
from java.io         import ByteArrayOutputStream

from jfig.utils      import SetupManager

from hades.simulator import SimObject
from hades.gui       import Editor
from hades.manager   import DesignManager
from hades.symbols   import SymbolManager
from hades.models    import StdLogic1164
from hades.models    import Const1164
from hades.signals   import Signal



CREATE_EDITOR_WAIT_MILLIS = 2000     # msecs. to wait when creating the Editor

editor = None
canvas = None
window = None



def checkCreateHadesEditor():
	global editor
	global window
	global canvas
	#
	# check if we already have a Hades editor, create it when not
	# Note that we also want to initialize the editor SetupManager
	#
	if (editor == None):
		SetupManager.loadGlobalProperties( "hades/hades.cnf" )
		SetupManager.loadUserProperties( "hades.cnf" )
		SetupManager.loadLocalProperties( "hades.cnf" )
		editor = Editor()
		# creating the editor takes some time...
		Thread.currentThread().sleep( CREATE_EDITOR_WAIT_MILLIS )
		print "pass: created the Hades editor"
	window = editor.getEditFrame()
	canvas = editor.getObjectCanvas()
	return



def startNewDesign():
	editor.doStartNewDesign()
	Thread.currentThread().sleep( 100 )
	return



def createSimObject( objectClassName ):
	target = DesignManager.getDesignManager().getSimObject( objectClassName )
	if (target == None):
		raise error, "FAIL: Could not instantiate class:", objectClassName
		return

	target.setEditor( editor )
	target.setVisible( 1 )
	symbol = SymbolManager.getSymbolManager().getSymbol( target )
	target.setSymbol( symbol )

	editor.getDesign().addComponent( target )
	target.getSymbol().setTrafo( canvas.getTrafo() )
	target.getSymbol().move( 1200, 1200 )
	editor.insertIntoObjectList( symbol )
	canvas.doFullRedraw()

	canvas.doFullRedraw()
	print "pass: getFullName: ", target.getFullName()
	return target


def checkPorts( simObject ):
	OK = 1
	ports = simObject.getPorts()
	# print "    : checkPorts: found", len(ports), "ports, now checking..."
	symbol = simObject.getSymbol()
	if (symbol == None):
		print "FAIL: Symbol is null in checkPorts"
		return
	for i in range( len(ports) ):
		portName = ports[i].getName()
		# print "...checking ", portName
		portPos = symbol.getPortPosition( portName )
		if (portPos == None):
			# special exception, e.g. in the hades.models.ttl74 classes:
			# "dummy" is used for ports[0] to get better pin-numbers
			if (portName != "dummy"):	
				print "FAIL: port ", portName, "not found on the symbol"
				OK = 0
	if (OK):
		print "pass: checkPorts"
	else:
		print "FAIL: checkPorts"
	return


def checkMoveObject( simObject ):
	symbol = simObject.getSymbol()
	pos    = symbol.getPos()
	bbox   = symbol.getBbox()

  # try to move 3600 units right, 4200 units down
	symbol.move( 3600, 4200 )
	pos2   = symbol.getPos()
	bbox2  = symbol.getBbox()

	OK = 1
	if ((pos2.x != pos.x + 3600) or (pos2.y != pos.y + 4200)):
		print "FAIL: symbol move position mismatch", pos2, "vs.", pos 
		OK = 0

	if (   (bbox2.getXl() != bbox.getXl() + 3600) 
	 		or (bbox2.getXr() != bbox.getXr() + 3600)
			or (bbox2.getYt() != bbox.getYt() + 4200)
			or (bbox2.getYb() != bbox.getYb() + 4200)):
		print "FAIL: symbol move bbox mismatch", bbox2, "vs.", bbox
		OK = 0

	#if (OK):
	#	print "pass: checkMoveObject"
	return
	


def copyObject( simObject ):
	oldPos = simObject.getSymbol().getPos()
	newPos = Point( oldPos.x + 7200, oldPos.y + 1200 )
	# copy object together with all the symbol stuff

	clone = DesignManager.getDesignManager().getCopy( simObject, editor );
	clone.setSymbol( SymbolManager.getSymbolManager().getSymbol(clone));
	clone.getSymbol().setTrafo( canvas.getTrafo() )
	clone.getSymbol().move( newPos.x,  newPos.y )

	editor.getDesign().addComponent( clone )
	editor.insertIntoObjectList( clone.getSymbol() )
	canvas.doFullRedraw()
	# print "pass: copyObject"
	return clone


def deleteObject( simObject ):
	editor.getDesign().deleteComponent( simObject )
	editor.getSimulator().deleteAllEventsFromSource( simObject )
	if (simObject.getPropertySheet() != None):
		simObject.getPropertySheet().setVisible( 0 )
	if (simObject.getSymbol() != None):
		editor.getObjectList().delete( simObject.getSymbol() )
		canvas.doFullRedraw()
	# print "pass: deleteObject"
	return


	
def checkMoveCopyDelete( simObject ):
	# move object's symbol
	checkMoveObject( simObject )
	# make three copies
	clone1 = copyObject( simObject )
	clone2 = copyObject( clone1 )
	clone3 = copyObject( clone2 )
	# delete copies
	deleteObject( clone2 )
	deleteObject( clone1 )
	deleteObject( clone3 )
	print "pass: checkMoveCopyDelete"




def checkConfigure( simObject ):
	simObject.configure()
	Thread.currentThread().sleep( 100 ) 
	dialog = simObject.getPropertySheet()
	if (dialog == None):
		print "FAIL: getPropertySheet returned null object!" 
		return
	else:
		dialog.setVisible( 0 )
	print "pass: checkConfigure"
	return


def checkWriteInitialize( simObject ):
	baos = ByteArrayOutputStream()
	pw = PrintWriter( baos )
	simObject.write( pw )
	pw.flush()
	print "pass: baos=", baos.toString()

	clone4 = copyObject( simObject )
	clone4.initialize( baos.toString() )
	print "pass: checkWriteInitialize"
	return




def testSimObject( objectClassName ):
	checkCreateHadesEditor()
	startNewDesign()	
	print "EXEC: testSimObject", objectClassName
	target = createSimObject( objectClassName )
	checkPorts( target )
	checkMoveCopyDelete( target )
	checkConfigure( target )
	checkWriteInitialize( target )
	# print "pass: testSimObject", objectClassName
	return  

	

# now run a few dummy tests
def test():
	testSimObject( "hades.models.gates.And2" )
	testSimObject( "hades.models.gates.Nand4" )
	testSimObject( "hades.models.flipflops.Dffre" )
	testSimObject( "hades.models.rtl.ROM_64Kx8" )
