   package mars.venus;
   import mars.*;
   import mars.mips.hardware.*;
   import mars.mips.instructions.*;
   import javax.swing.*;
   import java.awt.*;
   import java.awt.event.*;
   import java.util.*;   
   import javax.swing.table.*;
   import javax.swing.event.*;
	
	/*
Copyright (c) 2003-2006,  Pete Sanderson and Kenneth Vollmar

Developed by Pete Sanderson (psanderson@otterbein.edu)
and Kenneth Vollmar (kenvollmar@missouristate.edu)

Permission is hereby granted, free of charge, to any person obtaining 
a copy of this software and associated documentation files (the 
"Software"), to deal in the Software without restriction, including 
without limitation the rights to use, copy, modify, merge, publish, 
distribute, sublicense, and/or sell copies of the Software, and to 
permit persons to whom the Software is furnished to do so, subject 
to the following conditions:

The above copyright notice and this permission notice shall be 
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR 
ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF 
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

(MIT license, http://www.opensource.org/licenses/mit-license.html)
 */

  /**
    *  Creates the Text Segment window in the Execute tab of the UI
	 *   @author Team JSpim
	 **/  
	 
    public class TextSegmentWindow extends JInternalFrame{
      private  JTable table;
      private  Object[][] data;
   	/* Maintain an int array of code addresses in parallel with ADDRESS_COLUMN,
   	 * to speed model-row -> text-address mapping.  Maintain a Hashtable of
		 * (text-address, model-row) pairs to speed text-address -> model-row mapping.
		 * The former is used for breakpoints and changing display base (e.g. base 10 
		 * to 16); the latter is used for highlighting.  Both structures will remain
   	 * consistent once set up, since address column is not editable.
   	 */
      private  int[] intAddresses;      // index is table model row, value is text address
      private  Hashtable addressRows;   // key is text address, value is table model row
      private  Container contentPane;
      private  TextTableModel tableModel;
      private Font tableCellFont = new Font("Monospaced",Font.PLAIN,12);
      private  boolean codeHighlighting;
      private int highlightAddress;
      private TableModelListener tableModelListener;
   	
   	
      private static final String[] columnNames = {"Bkpt", "Address", "Code", "Basic", "Source"};
      private static final int BREAK_COLUMN = 0;
      private static final int ADDRESS_COLUMN = 1;
      private static final int CODE_COLUMN = 2;
      private static final int BASIC_COLUMN = 3;
      private static final int SOURCE_COLUMN = 4;
      
   	/**
   	  *  Constructor, sets up a new JInternalFrame.
   	  **/
   	
       public TextSegmentWindow(){
         super("Text Segment", true, false, true, true);
         contentPane = this.getContentPane();
         codeHighlighting = true;
               	
      }
   
     
   	/**
   	  *  Method to be called once the user compiles the program.  
   	  *  Should convert the lines of code over to the table rows and columns.
   	  **/
       public  void setupTable(){
         int addressBase = Globals.getGui().getMainPane().getExecutePane().getAddressDisplayBase();
         codeHighlighting = true;
         ArrayList sourceStatementList = Globals.program.getMachineList();
         data = new Object[sourceStatementList.size()][columnNames.length];
         intAddresses = new int[data.length];
         addressRows = new Hashtable(data.length);
      	
         for(int i=0; i < sourceStatementList.size(); i++){
            ProgramStatement statement = (ProgramStatement) sourceStatementList.get(i); 
            intAddresses[i] = statement.getAddress();
            addressRows.put(new Integer(intAddresses[i]),new Integer(i)); 
            data[i][BREAK_COLUMN]   = Boolean.FALSE;
            data[i][ADDRESS_COLUMN] = NumberDisplayBaseChooser.formatUnsignedInteger(statement.getAddress(), addressBase); 
            data[i][BASIC_COLUMN]   = statement.getBasicAssemblyStatement();
            data[i][SOURCE_COLUMN]  = ""+((statement.getSource().equals(""))? "" : statement.getSourceLine()+": "+statement.getSource());
            data[i][CODE_COLUMN]    = NumberDisplayBaseChooser.formatNumber(statement.getBinaryStatement(),16);
         }
         contentPane.removeAll();
         tableModel = new TextTableModel(data);
         if (tableModelListener!=null) {
            tableModel.addTableModelListener(tableModelListener);
            tableModel.fireTableDataChanged();// initialize listener
         }
         table= new MyTippedJTable(tableModel);
      	
      	// prevents cells in row from being highlighted when user clicks on breakpoint checkbox
         table.setRowSelectionAllowed(false);
      	
         table.getColumnModel().getColumn(BREAK_COLUMN).setMinWidth(40); 
         table.getColumnModel().getColumn(ADDRESS_COLUMN).setMinWidth(80); 
         table.getColumnModel().getColumn(CODE_COLUMN).setMinWidth(80);
      	
         table.getColumnModel().getColumn(BREAK_COLUMN).setMaxWidth(50); 
         table.getColumnModel().getColumn(ADDRESS_COLUMN).setMaxWidth(90); 
         table.getColumnModel().getColumn(CODE_COLUMN).setMaxWidth(90);
         table.getColumnModel().getColumn(BASIC_COLUMN).setMaxWidth(130);		
      
         table.getColumnModel().getColumn(BREAK_COLUMN).setPreferredWidth(40); 
         table.getColumnModel().getColumn(ADDRESS_COLUMN).setPreferredWidth(80); 
         table.getColumnModel().getColumn(CODE_COLUMN).setPreferredWidth(80);
         table.getColumnModel().getColumn(BASIC_COLUMN).setPreferredWidth(120);
         table.getColumnModel().getColumn(SOURCE_COLUMN).setPreferredWidth(300);
      	
         CodeCellRenderer codeStepHighlighter = new CodeCellRenderer(); 
         table.getColumnModel().getColumn(BASIC_COLUMN).setCellRenderer(codeStepHighlighter);
         table.getColumnModel().getColumn(SOURCE_COLUMN).setCellRenderer(codeStepHighlighter);
      	// to render String right-justified in mono font
         table.getColumnModel().getColumn(ADDRESS_COLUMN).setCellRenderer(new MonoRightCellRenderer());
         table.getColumnModel().getColumn(CODE_COLUMN).setCellRenderer(new MonoRightCellRenderer());
         contentPane.add(new JScrollPane(table, ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS, 
                         ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS));
      }
      
   	/**
   	 * remove all components
   	 */
       public void clearWindow() {
         contentPane.removeAll();
      }
   	
   	/**
   	 * Assign listener to Table model.  Used for breakpoints, since that is the only editable
   	 * column in the table.  Since table model objects are transient (get a new one with each 
   	 * successful assemble), this method will simply keep the identity of the listener then
   	 * add it as a listener each time a new table model object is created.  Limit 1 listener.
   	 */
       public void registerTableModelListener(TableModelListener tml) {
         tableModelListener = tml;
      } 
   	
    	/**
   	 *  Redisplay the addresses.  This should only be done when address display base is
   	 *  modified (e.g. between base 16 hex and base 10 dec).
   	 */
       public void updateCodeAddresses() {
         if (contentPane.getComponentCount() == 0) 
            return; // ignore if no content to change
         int addressBase = Globals.getGui().getMainPane().getExecutePane().getAddressDisplayBase();
         int address;
         String formattedAddress;
         for (int i=0; i<intAddresses.length; i++) {
            formattedAddress = NumberDisplayBaseChooser.formatUnsignedInteger(intAddresses[i], addressBase);
            table.getModel().setValueAt(formattedAddress, i, ADDRESS_COLUMN);
         }
      }
   	
   	/**
   	 *  Return code address as an int, for the specified row of the table.  This should only
   	 *  be used by the code renderer so I will not verify row.
   	 */
       int getIntCodeAddressAtRow(int row) {
         return intAddresses[row];
      }
   	
   	/**
   	 *  Returns number of breakpoints currently set.
   	 *  @return number of current breakpoints
   	 */
   	 
       public int getBreakpointCount() {
         int breakpointCount = 0;
         for(int i=0; i < data.length; i++){
            if (((Boolean)data[i][BREAK_COLUMN]).booleanValue()) {
               breakpointCount++;
            }
         }		 
         return breakpointCount;
      }
   	
   	/**
   	 *  Returns array of current breakpoints, each represented by a MIPS program counter address.
   	 *  These are stored in the BREAK_COLUMN of the table model.
   	 *  @return int array of breakpoints, sorted by PC address, or null if there are none.
   	 */
       public int[] getSortedBreakPointsArray() {
         int breakpointCount = getBreakpointCount();
         if (breakpointCount == 0) {
            return null;
         }
         int[] breakpoints = new int[breakpointCount];
         breakpointCount = 0;
         for(int i=0; i < data.length; i++){
            if (((Boolean)data[i][BREAK_COLUMN]).booleanValue()) {
               breakpoints[breakpointCount++] = intAddresses[i];
            }
         }
         Arrays.sort(breakpoints);
         return breakpoints;
      }
   	
   	/**
   	 * Clears all breakpoints that have been set since last assemble, and
   	 * updates the display of the breakpoint column.
   	 */
       public void clearAllBreakpoints() {
         for(int i=0; i < tableModel.getRowCount(); i++){
            if (((Boolean)data[i][BREAK_COLUMN]).booleanValue())  {
            // must use this method to assure display updated and listener notified
               tableModel.setValueAt(Boolean.FALSE, i, BREAK_COLUMN);
            }         
         }
      	// Handles an obscure situation: if you click to set some breakpoints then "immediately" clear them
      	// all using the shortcut (CTRL-K), the last checkmark set is not removed even though the breakpoint
      	// is removed (tableModel.setValueAt(Boolean.FALSE, i, BREAK_COLUMN)) and all the other checkmarks 
      	// are removed.  The checkmark remains although if you subsequently run the program it will blow 
      	// through because the data model cell really has been cleared (contains false).  Occurs only when 
      	// the last checked breakpoint check box still has the "focus".  There is but one renderer and editor 
      	// per column.  Getting the renderer and setting it "setSelected(false)" will not work.  You have 
      	// to get the editor instead.  (PS, 7 Aug 2006)
         ((JCheckBox)((DefaultCellEditor)table.getCellEditor(0,BREAK_COLUMN)).getComponent()).setSelected(false);
      }
   	
   	
   	/**
   	 *  Highlights the source code line whose address matches the current
   	 *  program counter value.  This is used for stepping through code
   	 *  execution and when reaching breakpoints.
   	 */
       public void highlightStepAtPC() {
         highlightStepAtAddress(RegisterFile.getProgramCounter());
      }
   
   	/**
   	 *  Highlights the source code line whose address matches the given
   	 *  text segment address.  
   	 *
   	 *  @param address  text segment address of instruction to be highlighted.
   	 */
   	 		 
       public void highlightStepAtAddress(int address) {
         highlightAddress = address;
         int highlightRow = 0;
         try {
            highlightRow = ((Integer)addressRows.get(new Integer(address))).intValue();
         } 
             catch (NullPointerException e) {
               return;// if address not in map, do nothing.
            }
         // Scroll if necessary to assure highlighted row is visible.
         int height = table.getRowHeight(highlightRow);
         int width = table.getColumnModel().getColumn(BASIC_COLUMN).getWidth() +
                        table.getColumnModel().getColumn(SOURCE_COLUMN).getWidth();
         int x = table.getColumnModel().getColumn(BREAK_COLUMN).getWidth() +
                    table.getColumnModel().getColumn(ADDRESS_COLUMN).getWidth();
         int y = height * highlightRow;
         table.scrollRectToVisible(new Rectangle(x,y,width,height));
         // Trigger highlighting, which is done by the column's cell renderer.
         // IMPLEMENTATION NOTE: Pretty crude implementation; mark all rows 
         // as changed so assure that the previously highlighted row is 
         // unhighlighted.  Would be better to keep track of previous row
         // then fire two events: one for it and one for the new row.
         table.tableChanged(new TableModelEvent(tableModel,0,data.length-1, BASIC_COLUMN));
         table.tableChanged(new TableModelEvent(tableModel,0,data.length-1, SOURCE_COLUMN));
      }
   
   /** Used to enable or disable source code highlighting.  If true (normally while
   *  stepping through execution) then MIPS statement at current program counter
   *  is highlighted.  The code column's cell renderer tests this variable.  
   */
       public void setCodeHighlighting(boolean set) {
         codeHighlighting = set;
      }
   
       public boolean getCodeHighlighting() {
         return codeHighlighting;
      }
   
   /** If any steps are highlighted, this erases the highlighting.
   *
   */
       public void unhighlightAllSteps() {
         boolean saved = this.getCodeHighlighting();
         this.setCodeHighlighting(false);
         table.tableChanged(new TableModelEvent(tableModel,0,data.length-1, BASIC_COLUMN));
         table.tableChanged(new TableModelEvent(tableModel,0,data.length-1, SOURCE_COLUMN));
         this.setCodeHighlighting(saved);
      }
     
     
     
   /** Inner class to implement the Table model for this JTable.
    */
       class TextTableModel extends AbstractTableModel {
         Object[][] data;
         
          public TextTableModel(Object[][] d){
            data=d;
         }
      
          public int getColumnCount() {
            return columnNames.length;
         }
        
          public int getRowCount() {
            return data.length;
         }
      
          public String getColumnName(int col) {
            return columnNames[col];
         }
      
          public Object getValueAt(int row, int col) {
            return data[row][col];
         }
      
        /*
         * JTable uses this method to determine the default renderer/
         * editor for each cell.  If we didn't implement this method,
         * then the break column would contain text ("true"/"false"),
         * rather than a check box.
         */
          public Class getColumnClass(int c) {
            return getValueAt(0, c).getClass();
         }
      
        /*
         * Don't need to implement this method unless your table's
         * editable.  Only Column #1, the Breakpoint, can be edited.
         */
          public boolean isCellEditable(int row, int col) {
            //Note that the data/cell address is constant,
            //no matter where the cell appears onscreen.
            if (col == BREAK_COLUMN) { 
               return true;
            } 
            else {
               return false;
            }
         }
      
        /*
         * Don't need to implement this method unless your table's
         * data can change.
         */
          public void setValueAt(Object value, int row, int col) {
            data[row][col] = value;
            fireTableCellUpdated(row, col);
         }
      
      
          private void printDebugData() {
            int numRows = getRowCount();
            int numCols = getColumnCount();
         
            for (int i=0; i < numRows; i++) {
               System.out.print("    row " + i + ":");
               for (int j=0; j < numCols; j++) {
                  System.out.print("  " + data[i][j]);
               }
               System.out.println();
            }
            System.out.println("--------------------------");
         }
      }  
   
   
     /*  a custom table cell renderer that we'll use to highlight the current line of 
   	*  source code when executing using Step or breakpoint.
   	*/
       class CodeCellRenderer extends DefaultTableCellRenderer { 
       
          public Component getTableCellRendererComponent(JTable table, Object value, 
                            boolean isSelected, boolean hasFocus, int row, int column) {									 
            Component cell = super.getTableCellRendererComponent(table, value, 
                                    isSelected, hasFocus, row, column);
            cell.setFont(tableCellFont);
            TextSegmentWindow textSegment = Globals.getGui().getMainPane().getExecutePane().getTextSegmentWindow();
            boolean highlighting = textSegment.getCodeHighlighting();
            if (highlighting && textSegment.getIntCodeAddressAtRow(row) == highlightAddress) { 
               cell.setBackground( Color.yellow );
            } 
            else {	
               cell.setBackground( Color.white );
            }
            return cell;
         }  
      
      }
       ///////////////////////////////////////////////////////////////////
   	 //
   	 // JTable subclass to provide custom tool tips for each of the
   	 // text table column headers. From Sun's JTable tutorial.
   	 // http://java.sun.com/docs/books/tutorial/uiswing/components/table.html
   	 //
       private class MyTippedJTable extends JTable {
          MyTippedJTable(TextTableModel m) {
            super(m);
         }       
         
         private String[] columnToolTips = {
               /* break */   "If checked, will set an execution breakpoint.",
               /* address */ "Text segment address of binary instruction code",
               /* code */    "32-bit binary MIPS instruction",
            	/* basic */   "Basic assembler instruction",
               /* source */  "Source code line"
               };
         	
          //Implement table header tool tips. 
          protected JTableHeader createDefaultTableHeader() {
            return 
                new JTableHeader(columnModel) {
                   public String getToolTipText(MouseEvent e) {
                     String tip = null;
                     java.awt.Point p = e.getPoint();
                     int index = columnModel.getColumnIndexAtX(p.x);
                     int realIndex = columnModel.getColumn(index).getModelIndex();
                     return columnToolTips[realIndex];
                  }
               };
         }
      }
      
   }
