   package mars.venus;
   import mars.*;
   import javax.swing.*;
   import javax.swing.text.*;
   import java.awt.*;
   import java.util.*;
   import java.awt.event.*;

/*
Copyright (c) 2003-2006,  Pete Sanderson and Kenneth Vollmar

Developed by Pete Sanderson (psanderson@otterbein.edu)
and Kenneth Vollmar (kenvollmar@missouristate.edu)

Permission is hereby granted, free of charge, to any person obtaining 
a copy of this software and associated documentation files (the 
"Software"), to deal in the Software without restriction, including 
without limitation the rights to use, copy, modify, merge, publish, 
distribute, sublicense, and/or sell copies of the Software, and to 
permit persons to whom the Software is furnished to do so, subject 
to the following conditions:

The above copyright notice and this permission notice shall be 
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR 
ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF 
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

(MIT license, http://www.opensource.org/licenses/mit-license.html)
 */

/**
  *  Creates the message window at the bottom of the UI.
  *   @author Team JSpim
  **/

    public class MessagesPane extends JTabbedPane{
      JTextArea assemble, run;
      JPanel assembleTab, runTab;
   	// These constants are designed to keep scrolled contents of the 
   	// two message areas from becoming overwhelmingly large (which
   	// seems to slow things down as new text is appended).  Once it
   	// reaches MAXIMUM_SCROLLED_CHARACTERS in length then cut off 
   	// the first NUMBER_OF_CHARACTERS_TO_CUT characters.  The latter
   	// must obviously be smaller than the former.
      public static final int MAXIMUM_SCROLLED_CHARACTERS = Globals.maximumMessageCharacters;
      public static final int NUMBER_OF_CHARACTERS_TO_CUT = Globals.maximumMessageCharacters/10 ; // 10%
   
   /**
     *  Constructor for the class, sets up two fresh tabbed text areas for program feedback.
     **/
   
       public MessagesPane() {
         super();
         this.setMinimumSize(new Dimension(0,0));
         assemble= new JTextArea();
         run= new JTextArea();
         assemble.setEditable(false); 
         run.setEditable(false);      	
      	
         JButton assembleTabClearButton = new JButton("Clear");
         assembleTabClearButton.setToolTipText("Clear the Mars Messages area");
         assembleTabClearButton.addActionListener(
                new ActionListener() {
                   public void actionPerformed(ActionEvent e){ 
                     assemble.setText("");
                  }
               });
         assembleTab = new JPanel(new BorderLayout());
         assembleTab.add(createBoxForButton(assembleTabClearButton),BorderLayout.WEST);
         assembleTab.add(new JScrollPane(assemble, ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED, 
                       ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED), BorderLayout.CENTER);
      					  
         JButton runTabClearButton = new JButton("Clear");
         runTabClearButton.setToolTipText("Clear the Run I/O area");
         runTabClearButton.addActionListener(
                new ActionListener() {
                   public void actionPerformed(ActionEvent e){ 
                     run.setText("");
                  }
               });
         runTab = new JPanel(new BorderLayout());
         runTab.add(createBoxForButton(runTabClearButton),BorderLayout.WEST);
         runTab.add(new JScrollPane(run, ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED, 
                       ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED), BorderLayout.CENTER);				
         this.addTab("Mars Messages", assembleTab);
         this.addTab("Run I/O", runTab);
         this.setToolTipTextAt(0,"Messages produced by Mars tools (\"Run\" menu)");
         this.setToolTipTextAt(1,"Simulated MIPS console input and output");
      }
		
      // Center given button in a box, centered vertically and 6 pixels on left and right
       private Box createBoxForButton(JButton button) {
         Box buttonRow = Box.createHorizontalBox();
         buttonRow.add(Box.createHorizontalStrut(6));
         buttonRow.add(button);
         buttonRow.add(Box.createHorizontalStrut(6));
         Box buttonBox = Box.createVerticalBox();
         buttonBox.add(Box.createVerticalGlue());
         buttonBox.add(buttonRow);
         buttonBox.add(Box.createVerticalGlue());
         return buttonBox;
      }
   	
   	/**
   	 * Returns component used to display assembler messages
   	 *
   	 * @return assembler message text component
   	 */
       public JTextArea getAssembleTextArea() {
         return assemble;
      }
   	/**
   	 * Returns component used to display runtime messages
   	 *
   	 * @return runtime message text component
   	 */   	
       public JTextArea getRunTextArea() {
         return run;
      }
   	
   	/**
   	 *  Post a message to the assembler display
   	 *
   	 *  @param message String to append to assembler display text
   	 */
       public void postMarsMessage(String message) {
         assemble.append(message);
      	// can do some crude cutting here.  If the document gets "very large", 
      	// let's cut off the oldest text. This will limit scrolling but the limit 
      	// can be set reasonably high.
         if (assemble.getDocument().getLength() > MAXIMUM_SCROLLED_CHARACTERS) {
            try {
               assemble.getDocument().remove(0, NUMBER_OF_CHARACTERS_TO_CUT);
            } 
                catch (BadLocationException ble) { 
               		   // only if NUMBER_OF_CHARACTERS_TO_CUT > MAXIMUM_SCROLLED_CHARACTERS
               }
         }
         setSelectedComponent(assembleTab);
      }
   	
      /**
   	 *  Post a message to the runtime display
   	 *
   	 *  @param message String to append to runtime display text
   	 */
   	// The work of this method is done by "invokeLater" because
   	// its JTextArea is maintained by the main event thread
   	// but also used, via this method, by the execution thread for 
   	// "print" syscalls. "invokeLater" schedules the code to be
   	// run under the event-processing thread no matter what.
   	// DPS, 23 Aug 2005.
       public void postRunMessage(String message) {
         final String mess = message;
         SwingUtilities.invokeLater(
                new Runnable() { 
                   public void run() { 
                     setSelectedComponent(runTab);
                     run.append(mess);
                  // can do some crude cutting here.  If the document gets "very large", 
                  // let's cut off the oldest text. This will limit scrolling but the limit 
                  // can be set reasonably high.
                     if (run.getDocument().getLength() > MAXIMUM_SCROLLED_CHARACTERS) {
                        try {
                           run.getDocument().remove(0, NUMBER_OF_CHARACTERS_TO_CUT);
                        } 
                            catch (BadLocationException ble) { 
                           // only if NUMBER_OF_CHARACTERS_TO_CUT > MAXIMUM_SCROLLED_CHARACTERS
                           }
                     }
                  } 
               });
      }
   	
   	/**
   	 * Make the assembler message tab current (up front)
   	 */
       public void selectMarsMessageTab() {
         setSelectedComponent(assembleTab);
      }
   	/**
   	 * Make the runtime message tab current (up front)
   	 */   	
       public void selectRunMessageTab() {
         setSelectedComponent(runTab);
      }
   
   	/**
   	 *  Method used by the SystemIO class to get interactive user input
   	 *  requested by a running MIPS program (e.g. syscall #5 to read an
   	 *  integer).  SystemIO knows whether simulator is being run at 
   	 *  command line by the user, or by the GUI. If run at command line, 
   	 *  it gets input from System.in rather than here.
		 *  @param prompt Prompt to display to the user.
		 *  @return User input.
   	 */
   	 /* Implementation note: I initially worked at having user input go
   	    directly into the Run I/O text area.  This is much more difficult,
   		 involving KeyListeners and FocusListeners and possibly threads.
   		 It is incredibly simpler just to pop up the input dialog then
   		 echo the input into the Run I/O text area.  
   	 */
       public String getInputString(String prompt) {
         String input;
      	// Statement shown on the next line is simpler, but I didn't want dialog to have Cancel button...
         //input = JOptionPane.showInputDialog(Globals.getGui(), prompt);
         JOptionPane pane = new JOptionPane(prompt,JOptionPane.QUESTION_MESSAGE,JOptionPane.DEFAULT_OPTION); 
         pane.setWantsInput(true);
         JDialog dialog = pane.createDialog(Globals.getGui(), "MIPS Keyboard Input"); 
         dialog.setVisible(true); 
         input = (String) pane.getInputValue();
         this.postRunMessage(Globals.userInputAlert+input+"\n");
         return input;
      }
   
   }