   package mars.venus;
   import mars.*;
   import mars.assembler.*;
   import java.io.*;
   import javax.swing.*;
   import java.awt.*;
   import java.awt.event.*;
   import java.util.*;
   import javax.swing.table.*;
   import javax.swing.border.*;
   
	/*
Copyright (c) 2003-2006,  Pete Sanderson and Kenneth Vollmar

Developed by Pete Sanderson (psanderson@otterbein.edu)
and Kenneth Vollmar (kenvollmar@missouristate.edu)

Permission is hereby granted, free of charge, to any person obtaining 
a copy of this software and associated documentation files (the 
"Software"), to deal in the Software without restriction, including 
without limitation the rights to use, copy, modify, merge, publish, 
distribute, sublicense, and/or sell copies of the Software, and to 
permit persons to whom the Software is furnished to do so, subject 
to the following conditions:

The above copyright notice and this permission notice shall be 
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR 
ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF 
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

(MIT license, http://www.opensource.org/licenses/mit-license.html)
 */
	
	/**
	  *  Represents the Labels window, which is a type of JInternalFrame.  Venus user
	  *  can view MIPS program labels.
	  *   @author Sanderson and Team JSpim
	  **/
    

	 

    public class LabelsWindow extends JInternalFrame{
      private Container contentPane;
      private JPanel labelPanel;      // holds J
      private JCheckBox dataLabels, textLabels;
      private ArrayList listOfLabelsForSymbolTable;
      private LabelsWindow labelsWindow;
      private static final int PREFERRED_NAME_COLUMN_WIDTH = 60;
      private static final int PREFERRED_ADDRESS_COLUMN_WIDTH = 60;
      private static final int MAX_DISPLAYED_CHARS = 24;
   
   
    /**
      *  Constructor for the Labels (symbol table) window.
   	**/
   
       public LabelsWindow (){
         super("Labels", true, false, true, true);
         labelsWindow = this;
         contentPane = this.getContentPane();
         labelPanel = new JPanel(new GridLayout(1,2,10,0));
         JPanel features = new JPanel();
         dataLabels = new JCheckBox("Data", true);
         textLabels = new JCheckBox("Text", true);
         dataLabels.addItemListener(new LabelItemListener());
         textLabels.addItemListener(new LabelItemListener());
         dataLabels.setToolTipText("If checked, will display labels defined in data segment");
         textLabels.setToolTipText("If checked, will display labels defined in text segment");
         features.add(dataLabels);
         features.add(textLabels);
         contentPane.add(features, BorderLayout.SOUTH);
         contentPane.add(labelPanel);
      }  
   	
   	/**
   	 * Initialize table of labels (symbol table)
   	 */
       public void setupTable(){
         labelPanel.removeAll();
         labelPanel.add(generateLabelScrollPane());
      }
   	
      /**
   	 * Clear the window
   	 */
       public void clearWindow() {
         labelPanel.removeAll();
      }
   	
   	//
       private JScrollPane generateLabelScrollPane() {
         listOfLabelsForSymbolTable = new ArrayList();
         listOfLabelsForSymbolTable.add(new LabelsForSymbolTable(null));// global symtab
         ArrayList MIPSprogramsAssembled = RunAssembleAction.getMIPSprogramsToAssemble();
         Box allSymtabTables = Box.createVerticalBox();
         for (int i=0; i<MIPSprogramsAssembled.size(); i++) {
            listOfLabelsForSymbolTable.add(new LabelsForSymbolTable(
                        (MIPSprogram) MIPSprogramsAssembled.get(i)));
         }
         ArrayList tableNames = new ArrayList();
         JTableHeader tableHeader = null;
         for (int i=0; i<listOfLabelsForSymbolTable.size(); i++) {
            LabelsForSymbolTable symtab = (LabelsForSymbolTable)listOfLabelsForSymbolTable.get(i);
            if (symtab.hasSymbols()) {
               String name = symtab.getSymbolTableName();
               if (name.length() > MAX_DISPLAYED_CHARS) {
                  name = name.substring(0,MAX_DISPLAYED_CHARS-3)+"...";
               }
            	// To get left-justified, put file name into first slot of horizontal Box, then glue.
               JLabel nameLab = new JLabel(name,JLabel.LEFT);
               Box nameLabel = Box.createHorizontalBox();
               nameLabel.add(nameLab);
               nameLabel.add(Box.createHorizontalGlue());
               nameLabel.add(Box.createHorizontalStrut(1));
               tableNames.add(nameLabel);
               allSymtabTables.add(nameLabel);
               JTable table = symtab.generateLabelTable();
               tableHeader = table.getTableHeader();
               allSymtabTables.add(table);
            }
         }
         JScrollPane labelScrollPane = new JScrollPane(allSymtabTables,
                                               ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS, 
            											  ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);
      	// Set file name label's max width to scrollpane's viewport width, max height to small.
      	// Does it do any good?  Addressing problem that occurs when label (filename) is wider than
      	// the table beneath it -- the table column widths are stretched to attain the same width and
      	// the address information requires scrolling to see.  All because of a long file name.
         for (int i=0;i<tableNames.size(); i++) {
            JComponent nameLabel = (JComponent)tableNames.get(i);
            nameLabel.setMaximumSize(new Dimension(
               labelScrollPane.getViewport().getViewSize().width,
               (int) (1.5*nameLabel.getFontMetrics(nameLabel.getFont()).getHeight())));
         }
         labelScrollPane.setColumnHeaderView(tableHeader);
         return labelScrollPane;
      }
   	
      /**
       * Method to update display of label addresses.  Since label information doesn't change,
       * this should only be done when address base is changed.
       *  (e.g. between base 16 hex and base 10 dec).
       */
       public void updateLabelAddresses() {
         if (listOfLabelsForSymbolTable != null) {
            for (int i=0; i<listOfLabelsForSymbolTable.size(); i++) {
               ((LabelsForSymbolTable)listOfLabelsForSymbolTable.get(i)).updateLabelAddresses();
            }      
         }
      }
   	
      
      ///////////////////////////////////////////////////////////////
   	//   Listener class to respond to "Text" or "Data" checkbox click 
       private class LabelItemListener implements ItemListener {
          public void itemStateChanged(ItemEvent ie) {
            for (int i=0; i<listOfLabelsForSymbolTable.size(); i++) {
               ((LabelsForSymbolTable)listOfLabelsForSymbolTable.get(i)).generateLabelTable();
            }     			 
         }
      }
      
   		 
   ///////////////////////////////////////////////////////////////////
   // Represents one symbol table for the display.
       private class LabelsForSymbolTable {
         private MIPSprogram myMIPSprogram;
         private Object[][] labelData;
         private JTable labelTable;	 
         private ArrayList symbols;
         private SymbolTable symbolTable;
         private String tableName;
      	
      	// Associated MIPSprogram object.  If null, this represents global symbol table.
          public LabelsForSymbolTable(MIPSprogram myMIPSprogram) {
            this.myMIPSprogram = myMIPSprogram;
            symbolTable = (myMIPSprogram == null)
                           ? Globals.symbolTable
               				: myMIPSprogram.getLocalSymbolTable();
            tableName = (myMIPSprogram == null)
                         ? "(global)"
               	       : new File(myMIPSprogram.getFilename()).getName();
         }
      	 
      	 // Returns file name of associated file for local symbol table or "(global)"    
          public String getSymbolTableName() {
            return tableName;
         }
      	
          public boolean hasSymbols() {
            return symbolTable.getSize()!=0;
         }
      
          // builds the Table containing labels and addresses for this symbol table.
          private JTable generateLabelTable() {
            String [] names= {"Label", "Address"};
            SymbolTable symbolTable = (myMIPSprogram == null)
                                      ? Globals.symbolTable
               							  : myMIPSprogram.getLocalSymbolTable();
            int addressBase = Globals.getGui().getMainPane().getExecutePane().getAddressDisplayBase();
            if (textLabels.isSelected() && dataLabels.isSelected()) {
               symbols = symbolTable.getAllSymbols();
            } 
            else if (textLabels.isSelected() && !dataLabels.isSelected()) {
               symbols = symbolTable.getTextSymbols();
            } 
            else if (!textLabels.isSelected() && dataLabels.isSelected()) {
               symbols = symbolTable.getDataSymbols();
            } 
            else {
               symbols = new ArrayList();
            }
            labelData = new Object[symbols.size()][2];
         
            for(int i=0; i< symbols.size(); i++){//sets up the label table
               Symbol s = (Symbol)(symbols.get(i));
               labelData[i][0] = s.getName();
               labelData[i][1] = NumberDisplayBaseChooser.formatNumber(s.getAddress(), addressBase);
            }
            LabelTableModel m = new LabelTableModel(labelData, names);
            if (labelTable == null) {
               labelTable = new MyTippedJTable(m); //JTable();
            	//labelTable.setBorder(BorderFactory.createLineBorder(Color.black));
            } 
            else {
               labelTable.setModel(m);
            }
         //            labelTable.getColumnModel().getColumn(0).setPreferredWidth(PREFERRED_NAME_COLUMN_WIDTH);
         //            labelTable.getColumnModel().getColumn(1).setPreferredWidth(PREFERRED_ADDRESS_COLUMN_WIDTH);
            labelTable.getColumnModel().getColumn(1).setCellRenderer(new MonoRightCellRenderer());
            return labelTable;
         }
      
      
          public void updateLabelAddresses() {
            if (labelPanel.getComponentCount() == 0) 
               return; // ignore if no content to change
            int addressBase = Globals.getGui().getMainPane().getExecutePane().getAddressDisplayBase();
            int address;
            String formattedAddress;
            int numSymbols = (labelData==null) ? 0 : labelData.length;
            for(int i=0; i< numSymbols; i++) {
               address = ((Symbol)symbols.get(i)).getAddress();
               formattedAddress = NumberDisplayBaseChooser.formatNumber(address, addressBase);
               labelTable.getModel().setValueAt(formattedAddress, i, 1);
            }
         }
      }   
   	//////////////////////  end of LabelsForOneSymbolTable class //////////////////  
   
   
   	
      ///////////////////////////////////////////////////////////////      
      // Class representing label table data 
       class LabelTableModel extends AbstractTableModel {
         String[] columnNames;
         Object[][] data;         
         
          public LabelTableModel(Object[][] d, String [] n){
            data=d;
            columnNames= n;
         }
         
          public int getColumnCount() {
            return columnNames.length;
         }
         
          public int getRowCount() {
            return data.length;
         }
         
          public String getColumnName(int col) {
            return columnNames[col];
         }
         
          public Object getValueAt(int row, int col) {
            return data[row][col];
         }
         
         /*
         * JTable uses this method to determine the default renderer/
         * editor for each cell.  
         */
          public Class getColumnClass(int c) {
            return getValueAt(0, c).getClass();
         }
         
         /*
         * Don't need to implement this method unless your table's
         * data can change.
         */
          public void setValueAt(Object value, int row, int col) {
            data[row][col] = value;
            fireTableCellUpdated(row, col);
         }
         
          private void printDebugData() {
            int numRows = getRowCount();
            int numCols = getColumnCount();
            
            for (int i=0; i < numRows; i++) {
               System.out.print("    row " + i + ":");
               for (int j=0; j < numCols; j++) {
                  System.out.print("  " + data[i][j]);
               }
               System.out.println();
            }
            System.out.println("--------------------------");
         }
      }  
      
       ///////////////////////////////////////////////////////////////////
       //
       // JTable subclass to provide custom tool tips for each of the
       // label table column headers. From Sun's JTable tutorial.
       // http://java.sun.com/docs/books/tutorial/uiswing/components/table.html
       //
       private class MyTippedJTable extends JTable {
          MyTippedJTable(LabelTableModel m) {
            super(m);
         }       
         
         private String[] columnToolTips = {
               /* label */   "Programmer-defined label (identifier).",
               /* address */ "Text or data segment address at which label is defined."
               };
         	
          //Implement table header tool tips. 
          protected JTableHeader createDefaultTableHeader() {
            return 
                new JTableHeader(columnModel) {
                   public String getToolTipText(MouseEvent e) {
                     String tip = null;
                     java.awt.Point p = e.getPoint();
                     int index = columnModel.getColumnIndexAtX(p.x);
                     int realIndex = columnModel.getColumn(index).getModelIndex();
                     return columnToolTips[realIndex];
                  }
               };
         }
      }
   }