   package mars.venus;
   import mars.*;
   import javax.swing.*;
   import javax.swing.event.*;
   import java.awt.*;
   import java.awt.event.*;
   import javax.swing.undo.*;
   import java.text.*;
   import java.util.*;
   import java.io.*;

/*
Copyright (c) 2003-2006,  Pete Sanderson and Kenneth Vollmar

Developed by Pete Sanderson (psanderson@otterbein.edu)
and Kenneth Vollmar (kenvollmar@missouristate.edu)

Permission is hereby granted, free of charge, to any person obtaining 
a copy of this software and associated documentation files (the 
"Software"), to deal in the Software without restriction, including 
without limitation the rights to use, copy, modify, merge, publish, 
distribute, sublicense, and/or sell copies of the Software, and to 
permit persons to whom the Software is furnished to do so, subject 
to the following conditions:

The above copyright notice and this permission notice shall be 
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR 
ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF 
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

(MIT license, http://www.opensource.org/licenses/mit-license.html)
 */

	/**
	  *  Creates the tabbed areas in the UI and also created the internal windows that 
	  *  exist in them.
	  *   @author Sanderson and Bumgarner
	  **/

    public class EditPane extends JPanel {

      private JTextArea sourceCode;
		private Font textEditingFont;
 
      private VenusUI mainUI;
      private UndoManager undo;
      private String currentDirectoryPath;
      private JLabel caretPositionLabel;
      private JCheckBox showLineNumbers;
      private JLabel lineNumbers;
      private static final char newline = '\n';
      private static int count = 0;
   
    /**
      *  Constructor for the EditPane class. 
   	**/
   	
       public EditPane(VenusUI appFrame, JTextArea sourceEditArea){
         super(new BorderLayout());
         this.mainUI = appFrame;
         currentDirectoryPath = System.getProperty("user.dir","c:\\");
         undo = new UndoManager();
         mainUI.editor = new Editor(mainUI);
      
         this.sourceCode= mainUI.sourceEditArea;
			textEditingFont = new Font("Monospaced",Font.PLAIN,12);
         this.sourceCode.setFont(textEditingFont);
         this.sourceCode.setTabSize(4);
         this.sourceCode.setMargin(new Insets(0,3,3,3));
      	// If source code is modified, will set flag to trigger/request file save.
         sourceCode.getDocument().addDocumentListener(
                new DocumentListener() {
                   public void insertUpdate(DocumentEvent evt) {
						   FileStatus.setEdited(true);
							switch (FileStatus.get()) {
							   case FileStatus.NEW_NOT_EDITED:
							      FileStatus.set(FileStatus.NEW_EDITED);
									break;
							   case FileStatus.NEW_EDITED:
								   break;
								default:
								   FileStatus.set(FileStatus.EDITED);
							}
							mainUI.editor.setFrameTitle();
                     if (showingLineNumbers()) {
                        lineNumbers.setText(getLineNumbersList());
                     }
                  }
                   public void removeUpdate(DocumentEvent evt) {
                     this.insertUpdate(evt);
                  }
                   public void changedUpdate(DocumentEvent evt) {
                     this.insertUpdate(evt);
                  }
               });
      	// Needed to support unlimited undo/redo capability
         sourceCode.getDocument().addUndoableEditListener(
                new UndoableEditListener() {
                   public void undoableEditHappened(UndoableEditEvent e) {
                  //Remember the edit and update the menus.
                     undo.addEdit(e.getEdit());
                     mainUI.editUndoAction.updateUndoState();
                     mainUI.editRedoAction.updateRedoState();
                  }			
               });
      	
         sourceCode.getCaret().addChangeListener(
                new ChangeListener() {
                   public void stateChanged(ChangeEvent e) {
                     displayCaretPosition(convertStreamPositionToLineColumn(sourceCode.getCaretPosition()));
                  }
               });
         this.setSourceCode("",false);
         
      	
         lineNumbers = new JLabel();
         lineNumbers.setFont(sourceCode.getFont());
         //lineNumbers.setForeground(Color.RED);
      	//lineNumbers.setBackground(Color.WHITE);
         lineNumbers.setVerticalAlignment(JLabel.TOP);
         lineNumbers.setText("");
         lineNumbers.setVisible(false);
         showLineNumbers = new JCheckBox("Show Line Numbers");
			showLineNumbers.setToolTipText("If checked, will display line number for each line of text.");
         showLineNumbers.setEnabled(false);
			// Show line numbers by default.
			showLineNumbers.setSelected(true); 
			lineNumbers.setVisible(true);
			
      	// Listener fires when "Show Line Numbers" check box is clicked.
         showLineNumbers.addItemListener(
                new ItemListener() {
                   public void itemStateChanged(ItemEvent e) {
                     if (showLineNumbers.isSelected()) {
                        lineNumbers.setText(getLineNumbersList());
                        lineNumbers.setVisible(true);
                     } 
                     else {
                        lineNumbers.setVisible(false);
                     }
                  	// needed because caret disappears when checkbox clicked
                     setCursorVisible(true); 
                     sourceCode.requestFocusInWindow();
                  }
               });
         JPanel source = new JPanel(new BorderLayout());
         source.add(lineNumbers,BorderLayout.WEST);
         source.add(sourceCode,BorderLayout.CENTER);
			JScrollPane editAreaScrollPane = new JScrollPane(source, 
			              ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS, 
                       ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS);
			editAreaScrollPane.getVerticalScrollBar().setUnitIncrement(
			              sourceCode.getFontMetrics(textEditingFont).getHeight());
			this.add(editAreaScrollPane, BorderLayout.CENTER);
         JPanel editInfo = new JPanel(new BorderLayout());
         caretPositionLabel = new JLabel();
			caretPositionLabel.setToolTipText("Tracks the current position of the text editing cursor.");
         displayCaretPosition(new Point());
         editInfo.add(caretPositionLabel,BorderLayout.WEST);
         editInfo.add(showLineNumbers,BorderLayout.CENTER);
         this.add(editInfo,BorderLayout.SOUTH);
      }
      
   	/**
   	  *  For initalizing the source code when opening an ASM file
   	  *   @param s String set to null from JSPIMMenu.
   	  *   @param editable set true if code is editable else false
   	  **/
   	
       public void setSourceCode(String s, boolean editable){
         this.sourceCode.setText(s);
         this.sourceCode.setBackground( (editable)? Color.WHITE : Color.GRAY);
         this.sourceCode.setEditable(editable);  
         this.sourceCode.setEnabled(editable);
         this.sourceCode.getCaret().setVisible(editable);
         this.sourceCode.setCaretPosition(0);
         if (editable) this.sourceCode.requestFocusInWindow();
      }
      
   
   	/** Form string with source code line numbers.
   	 * Resulting string is HTML, for which JLabel will happily honor <br> to do
   	 * multiline label (it ignores '\n').  The line number list is a JLabel with
   	 * one line number per line.
   	 */
       public String getLineNumbersList() {
         StringBuffer lineNumberList = new StringBuffer("<html>");
         int lineCount = this.getSourceLineCount();
         for (int i=1; i<=lineCount;i++) {
            lineNumberList.append(""+i+"<br>");
         }
         lineNumberList.append("</html>");
         return lineNumberList.toString();
      }
   
   
      /** Calculate and return number of lines in source code text.
   	 * Do this by counting newline characters then adding one if last line does
   	 * not end with newline character.
   	 */
   	 
   	/*  IMPLEMENTATION NOTE:
   	 * Tried repeatedly to use StringTokenizer to count lines but got bad results 
   	 * on empty lines (consecutive delimiters) even when returning delimiter as token.
   	 * BufferedReader on StringReader seems to work better.
   	 */
       public int getSourceLineCount() {
         BufferedReader bufStringReader = new BufferedReader(new StringReader(sourceCode.getText()));
         int lineNums = 0;
         try {
            while (bufStringReader.readLine() != null) {
               lineNums++;
            }
         }
             catch (IOException e) {
            }
         return lineNums;
      }
   	
   	/**
   	  *  Adds the source code line by line.
   	  *   @param s A line of source code.
   	  **/
      
       public void append(String s){ 
         this.sourceCode.append(s);
         this.sourceCode.setCaretPosition(0);
      }
   
      /**
		 * Get source code text
		 *
		 * @return Sting containing source code
		 */
       public String getSource(){
         return sourceCode.getText();
      }
   	
   	/**
		 *  Get the current directory path
		 *
		 * @return String containing current directory pathname
		 */
       public String getCurrentDirectoryPath() {
         return currentDirectoryPath;
      }
   	
		/**
		 *  Set the current directory pathname
		 *  
		 * @param path the desired directory pathname
		 */
       public void setCurrentDirectoryPath(String path) {
         currentDirectoryPath = path;
      }
		
      /**
		 *  get the manager in charge of UnDo operations
		 *  @return the UnDo manager
		 */
       public UndoManager getUndoManager() {
         return undo;
      }
      /*       Note: these are invoked only when copy/cut/paste are used from the
   	               toolbar or menu or the defined menu Alt codes.  When
   						Ctrl-C, Ctrl-X or Ctrl-V are used, this code is NOT invoked
   						but the operation works correctly!
   				The "set visible" operations are used because clicking on the toolbar
   				icon causes both the selection highlighting AND the blinking cursor
   				to disappear!  This does not happen when using menu selection or 
   				Ctrl-C/X/V
   	*/
		/**
		 * copy currently-selected text into clipboard
		 */
       public void copyText() {
         sourceCode.copy();
         setCursorVisible(true);
         sourceCode.getCaret().setSelectionVisible(true);
      }
   	 
		/**
		 * cut currently-selected text into clipboard
		 */
		 public void cutText() {
         sourceCode.cut();		 
         setCursorVisible(true);
      }
 		/**
		 * paste clipboard contents at cursor position
		 */  
       public void pasteText() {
         sourceCode.paste();		 
         setCursorVisible(true);
      }		 
   	 
   	 /**
		  * Control cursor visibility
		  *
		  * @param vis true to display cursor, false to hide it
		  */
       public void setCursorVisible(boolean vis) {
         sourceCode.getCaret().setVisible(vis);
      }
   	 
   	/**
		 *  get editor's line number display status 
		 *
		 *  @return true if editor is current displaying line numbers, false otherwise.
		 */
       public boolean showingLineNumbers() {
         return showLineNumbers.isSelected();
      }

   	/**
		 *  enable or disable checkbox that controls display of line numbers 
		 *
		 *  @param enable True to enable box, false to disable.
		 */   	 
       public void setShowLineNumbersEnabled(boolean enabled) {
         showLineNumbers.setEnabled(enabled);
         //showLineNumbers.setSelected(false); // set off, whether closing or opening
      }
   	 /**
		  *  Display cursor coordinates
		  *
		  *  @param p  Point object with x-y (column, line number) coordinates of cursor
		  */
       public void displayCaretPosition(Point p) {
         caretPositionLabel.setText("Line: "+ p.y + " Column: "+ p.x);
      }
   	 
		 /**
		  * Given byte stream position in text being edited, calculate its column and line
		  * number coordinates.
		  *
		  * @param stream position of character
		  * @return position Its column and line number coordinate as a Point
		  */
       public Point convertStreamPositionToLineColumn(int position) {
         String textStream = sourceCode.getText();
         int line = 1;
         int column = 1;
         for (int i=0; i<position; i++) {
            if (textStream.charAt(i) == newline) {
               line++;
               column=1;
            } 
            else {
               column++;
            }
         }
         return new Point(column,line);
      }
   }