   package mars.venus;
   import mars.*;
   import mars.util.*;
   import mars.mips.hardware.*;
   import javax.swing.*;
   import java.awt.*;
   import java.awt.event.*;
   import java.util.*;
   import javax.swing.table.*;
   import javax.swing.border.*;

/*
Copyright (c) 2003-2006,  Pete Sanderson and Kenneth Vollmar

Developed by Pete Sanderson (psanderson@otterbein.edu)
and Kenneth Vollmar (kenvollmar@missouristate.edu)

Permission is hereby granted, free of charge, to any person obtaining 
a copy of this software and associated documentation files (the 
"Software"), to deal in the Software without restriction, including 
without limitation the rights to use, copy, modify, merge, publish, 
distribute, sublicense, and/or sell copies of the Software, and to 
permit persons to whom the Software is furnished to do so, subject 
to the following conditions:

The above copyright notice and this permission notice shall be 
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR 
ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF 
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

(MIT license, http://www.opensource.org/licenses/mit-license.html)
 */
   
	/**
	  *  Represents the Data Segment window, which is a type of JInternalFrame.
	  *   @author Sanderson and Bumgarner
	  **/
    
    public class DataSegmentWindow extends JInternalFrame {
   
      private static final String [] dataSegmentNames={"Data", "Stack", "Kernel"};
      private static  Object[][] dataData;
   
      private static JTable dataTable;
      private Container contentPane;
      private JPanel tablePanel;
      private JButton homeButton, nextButton, prevButton, stakButton, globButton, heapButton, kernButton, extnButton, mmioButton;
   
      static final int VALUES_PER_ROW = 8;
      static final int NUMBER_OF_ROWS = 16;  // with 8 value columns, this shows 512 bytes;  
      static final int NUMBER_OF_COLUMNS = VALUES_PER_ROW + 1;// 1 for address and 8 for values
      static final int BYTES_PER_VALUE = 4;
      static final int BYTES_PER_ROW = VALUES_PER_ROW * BYTES_PER_VALUE;
      static final int MEMORY_CHUNK_SIZE = NUMBER_OF_ROWS * BYTES_PER_ROW;
      static final int ADDRESS_COLUMN = 0;
      static final boolean USER_MODE = false;
      static final boolean KERNEL_MODE = true;
      int firstAddress;
      int homeAddress;
      boolean userOrKernelMode;
   
   	// The combo box replaced the row of buttons when number of buttons expanded to 7!
   	// We'll keep the button objects however and manually invoke their action listeners
   	// when the corresponding combo box item is selected.  DPS 22-Nov-2006
      JComboBox  baseAddressSelector;
   	
   	// The next bunch are initialized dynamically in initializeBaseAddressChoices()
      private String[] displayBaseAddressChoices; 
      private int[] displayBaseAddresses;
      private int defaultBaseAddressIndex;
      JButton[] baseAddressButtons;
   
   
    /**
      *  Constructor for the Data Segment window.
   	*   @param choosers an array of objects used by user to select number display base (10 or 16)
   	*/
   
       public DataSegmentWindow (NumberDisplayBaseChooser[] choosers){
         super("Data Segment", true, false, true, true);
         homeAddress = Globals.memory.dataBaseAddress;  // address for Home button
         firstAddress = homeAddress;  // first address to display at any given time
         userOrKernelMode = USER_MODE;
         contentPane = this.getContentPane();
         tablePanel = new JPanel(new GridLayout(1,2,10,0));
         JPanel features = new JPanel();
         Toolkit tk = Toolkit.getDefaultToolkit();
         Class cs = this.getClass(); 
         try {
            prevButton = new JButton(new ImageIcon(tk.getImage(cs.getResource(Globals.imagesPath+"Previous22.png"))));//"Back16.gif"))));//"Down16.gif"))));
            nextButton = new JButton(new ImageIcon(tk.getImage(cs.getResource(Globals.imagesPath+"Next22.png"))));//"Forward16.gif")))); //"Up16.gif"))));
            homeButton = new JButton(".data");//new ImageIcon(tk.getImage(cs.getResource("/images/Home16.gif"))));
            stakButton = new JButton("$sp");
            globButton = new JButton("$gp");
            heapButton = new JButton("heap");
            extnButton = new JButton(".extern");
            mmioButton = new JButton("MMIO");
            kernButton = new JButton(".kdata");
         } 
             catch (NullPointerException e) {
               System.out.println("Internal Error: images folder not found");
               System.exit(0);
            }
      	
         initializeBaseAddressChoices();      	      	
         baseAddressSelector = new JComboBox(displayBaseAddressChoices);
         baseAddressSelector.setEditable(false);
         baseAddressSelector.setSelectedIndex(defaultBaseAddressIndex);
         baseAddressSelector.setToolTipText("Base address for data segment display");
         baseAddressSelector.addActionListener(
                new ActionListener() {
                   public void actionPerformed(ActionEvent e) {
                   // trigger action listener for associated invisible button.
                     baseAddressButtons[baseAddressSelector.getSelectedIndex()].getActionListeners()[0].actionPerformed(null);
                  }
               });
      			
         addButtonActionListenersAndInitialize();
         JPanel navButtons = new JPanel(new GridLayout(1,4));
         navButtons.add(prevButton);
         navButtons.add(nextButton);
      	/*  This group of buttons was replaced by a combo box.  Keep the JButton objects for their action listeners.
         navButtons.add(homeButton);
         navButtons.add(heapButton);
         navButtons.add(stakButton);
         navButtons.add(globButton);
         navButtons.add(extnButton);
         navButtons.add(kernButton);
         navButtons.add(mmioButton);//*/
         features.add(navButtons);
         features.add(baseAddressSelector);
         for (int i=0; i<choosers.length; i++) {
            features.add(choosers[i]);
         }
         contentPane.add(features, BorderLayout.SOUTH);  
      }  
   
   
       ////////////////////////////////////////////////////////////////////////
       // Initalize arrays used with Base Address combo box chooser.
   	 // The combo box replaced the row of buttons when number of buttons expanded to 7!
       private void initializeBaseAddressChoices() {
         int[] displayBaseAddressArray = {Memory.dataSegmentBaseAddress, -1 /*Memory.globalPointer*/, 
                                          Memory.dataBaseAddress, Memory.heapBaseAddress, 
               									-1 /*Memory.stackBaseAddress*/, Memory.kernelDataBaseAddress, 
               									Memory.memoryMapBaseAddress };
      	// Must agree with above in number and order...
         String[] descriptions =         { " (.extern)", "current $gp", " (.data)", " (heap)", 
                                           "current $sp"," (.kdata)"," (MMIO)" };
         // Also must agree in number and order.  Upon combo box item selection, will invoke
      	// action listener for that item's button.  
         baseAddressButtons = new JButton[descriptions.length];
         baseAddressButtons[0]=extnButton; baseAddressButtons[1]=globButton; baseAddressButtons[2]=homeButton;
         baseAddressButtons[3]=heapButton; baseAddressButtons[4]=stakButton; baseAddressButtons[5]=kernButton;
         baseAddressButtons[6]=mmioButton;
         displayBaseAddresses = displayBaseAddressArray;
         displayBaseAddressChoices = new String[displayBaseAddressArray.length];
         for (int i=0; i<displayBaseAddressChoices.length; i++) {
            displayBaseAddressChoices[i] =  ((displayBaseAddressArray[i] != -1)
                                            ? mars.util.Binary.intToHexString(displayBaseAddressArray[i])
               									  : "")
               									  +descriptions[i];
         }
         defaultBaseAddressIndex = 2;  // default to 0x10010000 (.data)
      }
      
   	
   	////////////////////////////////////////////////////////////////////////////////
   	//  Generates the Address/Data part of the Data Segment window.
   	//   Returns the JScrollPane for the Address/Data part of the Data Segment window.
       private JScrollPane generateDataPanel(){
         dataData = new Object[NUMBER_OF_ROWS][NUMBER_OF_COLUMNS];
         int valueBase = Globals.getGui().getMainPane().getExecutePane().getValueDisplayBase();
         int addressBase = Globals.getGui().getMainPane().getExecutePane().getAddressDisplayBase();
         int address = this.homeAddress;
         for(int row=0; row<NUMBER_OF_ROWS; row++){
            dataData[row][ADDRESS_COLUMN] = NumberDisplayBaseChooser.formatUnsignedInteger(address, addressBase);
            for (int column=1; column<NUMBER_OF_COLUMNS; column++) {
               try {
                  dataData[row][column] = NumberDisplayBaseChooser.formatNumber(Globals.memory.getRawWord(address), valueBase);
               } 
                   catch (AddressErrorException aee) {
                     dataData[row][column] = NumberDisplayBaseChooser.formatNumber(0, valueBase);
                  }
               address += BYTES_PER_VALUE;
            }
         }
         String [] names = new String[NUMBER_OF_COLUMNS];
         for (int i=0; i<NUMBER_OF_COLUMNS; i++) {
            names[i] = getHeaderStringForColumn(i, addressBase);
         }
         dataTable= new MyTippedJTable(new DataTableModel(dataData, names));
      	// Do not allow user to re-order columns; column order corresponds to MIPS memory order
         dataTable.getTableHeader().setReorderingAllowed(false);
      	// Addresses and values are Strings, render right-justified in mono font
         for (int i=0; i<NUMBER_OF_COLUMNS; i++) {
            dataTable.getColumnModel().getColumn(i).setPreferredWidth(60);
            dataTable.getColumnModel().getColumn(i).setCellRenderer(new MonoRightCellRenderer());
         }
         return new JScrollPane(dataTable,ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS, 
                                ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS);
      }
      
   	// Little helper.  Is called when headers set up and each time number base changes.
       private String getHeaderStringForColumn(int i, int base) {
         return (i==ADDRESS_COLUMN)? "Address" : "Value (+"+Integer.toString((i-1)*BYTES_PER_VALUE, base)+")";
      }
   	
   	
   	/**
   	 * Generates and displays fresh table, typically done upon successful assembly.
   	 */   	
       public void setupTable(){
         tablePanel.removeAll();
         tablePanel.add(generateDataPanel()); 
         contentPane.add(tablePanel);
         enableAllButtons();
      }
      
   	/**
   	 * Removes the table from its frame, typically done when a file is closed.
   	 */
       public void clearWindow() {
         tablePanel.removeAll();
         disableAllButtons();
      }
   
   	/**
   	 * Update table model with contents of new memory "chunk".  Mars supports megabytes of
   	 * data segment space so we only plug a "chunk" at a time into the table.
   	 * @param firstAddr the first address in the memory range to be placed in the model.
   	 */
   	 
       public void updateModelForMemoryRange(int firstAddr) {
         if (tablePanel.getComponentCount() == 0) 
            return; // ignore if no content to change
         int valueBase = Globals.getGui().getMainPane().getExecutePane().getValueDisplayBase();
         int addressBase = Globals.getGui().getMainPane().getExecutePane().getAddressDisplayBase();
         int address = firstAddr;
         TableModel dataModel = dataTable.getModel();
         for (int row=0; row<NUMBER_OF_ROWS; row++) {
            ((DataTableModel)dataModel).setDisplayAndModelValueAt(NumberDisplayBaseChooser.formatUnsignedInteger(address, addressBase),row,ADDRESS_COLUMN);
            for (int column=1; column<NUMBER_OF_COLUMNS; column++) {
               try {
                  ((DataTableModel)dataModel).setDisplayAndModelValueAt(NumberDisplayBaseChooser.formatNumber(Globals.memory.getWord(address), valueBase),row,column);
               } 
                   catch (AddressErrorException aee) {
                     ((DataTableModel)dataModel).setDisplayAndModelValueAt(NumberDisplayBaseChooser.formatNumber(0, valueBase),row,column);
                  }
               address += BYTES_PER_VALUE;
            }
         }
      }
   	 
   	/**
   	 * Update data display to show this value (I'm not sure it is being called).
   	 */
   	
       public void updateCell(int address, int value) {
         int offset = address-this.firstAddress;
         if (offset<0 || offset>=MEMORY_CHUNK_SIZE) { // out of range
            return;
         }
         int row = offset/BYTES_PER_ROW;
         int column = (offset % BYTES_PER_ROW)/BYTES_PER_VALUE + 1; // column 0 reserved for address
         int valueBase = Globals.getGui().getMainPane().getExecutePane().getValueDisplayBase();
         ((DataTableModel)dataTable.getModel()).setDisplayAndModelValueAt(NumberDisplayBaseChooser.formatNumber(value, valueBase),
                    row,column);
      }
   	
   	/**
   	 *  Redisplay the addresses.  This should only be done when address display base is
   	 *  modified (e.g. between base 16, hex, and base 10, dec).
   	 */
       public void updateDataAddresses() {
         if (tablePanel.getComponentCount() == 0) 
            return; // ignore if no content to change
         int addressBase = Globals.getGui().getMainPane().getExecutePane().getAddressDisplayBase();
         int address = this.firstAddress;
         String formattedAddress;
         for(int i=0; i<NUMBER_OF_ROWS; i++) {
            formattedAddress = NumberDisplayBaseChooser.formatUnsignedInteger(address, addressBase);
            ((DataTableModel)dataTable.getModel()).setDisplayAndModelValueAt(formattedAddress, i, 0);
            address += BYTES_PER_ROW;
         }
      	// column headers include address offsets, so translate them too
         for (int i=1; i<NUMBER_OF_COLUMNS; i++) {
            dataTable.getColumnModel().getColumn(i).setHeaderValue(getHeaderStringForColumn(i, addressBase));
         }
         dataTable.getTableHeader().repaint();
      }
   	
      /**
   	 * Update data display to show all values
   	 */
   	 
       public void updateValues(){
         updateModelForMemoryRange(this.firstAddress);
      }
   
      /**
   	 * Reset range of memory addresses to intial range and update display.
   	 */
   	 
       public void resetMemoryRange(){
         homeButton.getActionListeners()[0].actionPerformed(null);
      }
   
      /**
   	 * Reset all data display values to 0
   	 */   
   	
       public void resetValues(){
         int valueBase = Globals.getGui().getMainPane().getExecutePane().getValueDisplayBase();
         TableModel dataModel = dataTable.getModel();
         for (int row=0; row<NUMBER_OF_ROWS; row++) {
            for (int column=1; column<NUMBER_OF_COLUMNS; column++) {
               ((DataTableModel)dataModel).setDisplayAndModelValueAt(NumberDisplayBaseChooser.formatNumber(0, valueBase),row,column);
            }
         }
         disableAllButtons();
      }
   	
   	/*
   	 * Do this initially and upon reset.
   	 */
       private void disableAllButtons() {
         baseAddressSelector.setEnabled(false);
         globButton.setEnabled(false);
         stakButton.setEnabled(false);
         heapButton.setEnabled(false);
         extnButton.setEnabled(false);
         mmioButton.setEnabled(false);
         kernButton.setEnabled(false);
         prevButton.setEnabled(false);
         nextButton.setEnabled(false);
         homeButton.setEnabled(false);
      }
   
   	/*
   	 * Do this upon reset.
   	 */
       private void enableAllButtons() {
         baseAddressSelector.setEnabled(true);
         globButton.setEnabled(true);
         stakButton.setEnabled(true);
         heapButton.setEnabled(true);
         extnButton.setEnabled(true);
         mmioButton.setEnabled(true);
         kernButton.setEnabled(true);
         prevButton.setEnabled(true);
         nextButton.setEnabled(true);
         homeButton.setEnabled(true);
      }
   	
   	/*
   	 * Establish action listeners for the data segment navigation buttons.
   	 */
   	
       private void addButtonActionListenersAndInitialize() {
         // set initial states
         disableAllButtons();
      	// add tool tips
         globButton.setToolTipText("View range around global pointer");
         stakButton.setToolTipText("View range around stack pointer");
         heapButton.setToolTipText("View range around heap base address "+
                         Binary.intToHexString(Globals.memory.heapBaseAddress));
         kernButton.setToolTipText("View range around kernel data base address "+
                         Binary.intToHexString(Globals.memory.kernelDataBaseAddress));
         extnButton.setToolTipText("View range around static global base address "+
                         Binary.intToHexString(Globals.memory.dataSegmentBaseAddress));
         mmioButton.setToolTipText("View range around MMIO base address "+
                         Binary.intToHexString(Globals.memory.memoryMapBaseAddress));
         prevButton.setToolTipText("View next lower address range");
         nextButton.setToolTipText("View next higher address range");
         homeButton.setToolTipText("View range around static data segment base address "+
                         Binary.intToHexString(Globals.memory.dataBaseAddress));
      	// add the action listeners to maintain button state and table contents
      	// Currently there is no memory upper bound so next button always enabled.
      
         globButton.addActionListener( 
                new ActionListener() {
                   public void actionPerformed(ActionEvent ae) {
                     userOrKernelMode = USER_MODE;
                  	// get $gp global pointer, but guard against it having value below data segment
                     firstAddress = Math.max(Globals.memory.dataSegmentBaseAddress,RegisterFile.getValue(28)); 
                  	// updateModelForMemoryRange requires argument to be multiple of 4
                     firstAddress = firstAddress - (firstAddress % 4);
                     homeAddress = firstAddress; 
                     firstAddress = setFirstAddressAndPrevNextButtonEnableStatus(firstAddress);
                     updateModelForMemoryRange(firstAddress);
                  }
               });
      	
         stakButton.addActionListener( 
                new ActionListener() {
                   public void actionPerformed(ActionEvent ae) {
                     userOrKernelMode = USER_MODE;
                  	// get $sp stack pointer, but guard against it having value below data segment
                     firstAddress = Math.max(Globals.memory.dataSegmentBaseAddress,RegisterFile.getValue(29)); 
                  	// updateModelForMemoryRange requires argument to be multiple of 4
                     firstAddress = firstAddress - (firstAddress % 4);
                     homeAddress = Globals.memory.stackBaseAddress;
                     firstAddress = setFirstAddressAndPrevNextButtonEnableStatus(firstAddress);
                     updateModelForMemoryRange(firstAddress);
                  }
               });
      
         heapButton.addActionListener( 
                new ActionListener() {
                   public void actionPerformed(ActionEvent ae) {
                     userOrKernelMode = USER_MODE;
                     homeAddress = Globals.memory.heapBaseAddress;
                     firstAddress = setFirstAddressAndPrevNextButtonEnableStatus(homeAddress);
                     updateModelForMemoryRange(firstAddress);
                  }
               });
      
         extnButton.addActionListener( 
                new ActionListener() {
                   public void actionPerformed(ActionEvent ae) {
                     userOrKernelMode = USER_MODE;
                     homeAddress = Globals.memory.dataSegmentBaseAddress;
                     firstAddress = setFirstAddressAndPrevNextButtonEnableStatus(homeAddress);
                     updateModelForMemoryRange(firstAddress);
                  }
               }); 
      			     			
         kernButton.addActionListener( 
                new ActionListener() {
                   public void actionPerformed(ActionEvent ae) {
                     userOrKernelMode = KERNEL_MODE;
                     homeAddress = Globals.memory.kernelDataBaseAddress;
                     firstAddress = homeAddress;
                     firstAddress = setFirstAddressAndPrevNextButtonEnableStatus(firstAddress);
                     updateModelForMemoryRange(firstAddress);
                  }
               });
      			     			
         mmioButton.addActionListener( 
                new ActionListener() {
                   public void actionPerformed(ActionEvent ae) {
                     userOrKernelMode = KERNEL_MODE;
                     homeAddress = Globals.memory.memoryMapBaseAddress;
                     firstAddress = homeAddress;
                     firstAddress = setFirstAddressAndPrevNextButtonEnableStatus(firstAddress);
                     updateModelForMemoryRange(firstAddress);
                  }
               });      
      	
         homeButton.addActionListener( 
                new ActionListener() {
                   public void actionPerformed(ActionEvent ae) {
                     userOrKernelMode = USER_MODE;
                     homeAddress = Globals.memory.dataBaseAddress;
                     firstAddress = homeAddress;
                     firstAddress = setFirstAddressAndPrevNextButtonEnableStatus(firstAddress);
                     updateModelForMemoryRange(firstAddress);
                  }
               });      			
      			
         prevButton.addActionListener( 
                new ActionListener() {
                   public void actionPerformed(ActionEvent ae) {
                     firstAddress -= MEMORY_CHUNK_SIZE;
                     firstAddress = setFirstAddressAndPrevNextButtonEnableStatus(firstAddress);
                     updateModelForMemoryRange(firstAddress);
                  }
               });
      	
         nextButton.addActionListener( 
                new ActionListener() {
                   public void actionPerformed(ActionEvent ae) {
                     firstAddress += MEMORY_CHUNK_SIZE;
                     firstAddress = setFirstAddressAndPrevNextButtonEnableStatus(firstAddress);
                     updateModelForMemoryRange(firstAddress);
                  }
               });
      
      }
   	
   	////////////////////////////////////////////////////////////////////////////////////
   	// This will assure that user cannot view memory locations outside the data segment
   	// for selected mode.  For user mode, this means no lower than data segment base,
   	// or higher than user memory boundary.  For kernel mode, this means no lower than
   	// kernel data segment base or higher than kernel memory.  It is called by the
   	// above action listeners.
   	//
   	// lowAddress is lowest desired address to view, it is adjusted if necessary
   	// and returned.
   	//
   	// PrevButton and NextButton are enabled/disabled appropriately.
   	//
       private int setFirstAddressAndPrevNextButtonEnableStatus(int lowAddress) {
         int lowLimit = (userOrKernelMode==USER_MODE) ? Globals.memory.dataSegmentBaseAddress
                                                      : Globals.memory.kernelDataBaseAddress;
         int highLimit= (userOrKernelMode==USER_MODE) ? Globals.memory.userHighAddress
                                                      : Globals.memory.kernelHighAddress;
         if (lowAddress <= lowLimit) {
            lowAddress = lowLimit;
            prevButton.setEnabled(false);
         } 
         else {
            prevButton.setEnabled(true);
         }
         if (lowAddress >= highLimit - MEMORY_CHUNK_SIZE) {
            lowAddress = highLimit - MEMORY_CHUNK_SIZE + 1;
            nextButton.setEnabled(false);
         } 
         else {
            nextButton.setEnabled(true);
         }
         return lowAddress;
      }
   
   	
   	
   	////////////////////////////////////////////////////////////////////////
   	// Class representing memory data table data
   	
       class DataTableModel extends AbstractTableModel {
         String[] columnNames;
         Object[][] data;         
      	
          public DataTableModel(Object[][] d, String [] n){
            data=d;
            columnNames= n;
         }
      
          public int getColumnCount() {
            return columnNames.length;
         }
        
          public int getRowCount() {
            return data.length;
         }
      
          public String getColumnName(int col) {
            return columnNames[col];
         }
      
          public Object getValueAt(int row, int col) {
            return data[row][col];
         }
      
        /*
         * Don't need to implement this method unless your table's
         * editable.  
         */
          public boolean isCellEditable(int row, int col) {
            //Note that the data/cell address is constant,
            //no matter where the cell appears onscreen.
            if (col != ADDRESS_COLUMN) { 
               return true;
            } 
            else {
               return false;
            }
         }
      
      
      
        /*
         * JTable uses this method to determine the default renderer/
         * editor for each cell.  
         */
          public Class getColumnClass(int c) {
            return getValueAt(0, c).getClass();
         }
      
      
        /*
         * Update cell contents in table model.  This method should be called
      	* only when user edits cell, so input validation has to be done.  If
      	* value is valid, MIPS memory is updated.
         */
          public void setValueAt(Object value, int row, int col) {
            int val=0;
            int address=0;
            try {
               val = Binary.stringToInt((String) value);
            }
                catch (NumberFormatException nfe) {
                  data[row][col] = "INVALID";
                  fireTableCellUpdated(row, col);
                  return;
               }
         
               // calculate address from row and column
            try {
               address = Binary.stringToInt((String)data[row][ADDRESS_COLUMN]) + (col-1)*BYTES_PER_VALUE;  // KENV 1/6/05
            }
                catch (NumberFormatException nfe) {
                  //  can't really happen since memory addresses are completely under
                  // the control of my software.
               }
         	//  Assures that if changed during MIPS program execution, the update will
         	//  occur only between MIPS instructions.
            synchronized (Globals.memoryAndRegistersLock) {
               try {
                  Globals.memory.setRawWord(address,val);
               } 
                // somehow, user was able to display out-of-range address.  Most likely to occur between
                // stack base and Kernel.
                   catch (AddressErrorException aee) {
                     val = 0;
                  }
            }// end synchronized block
            int valueBase = Globals.getGui().getMainPane().getExecutePane().getValueDisplayBase();
            data[row][col] = NumberDisplayBaseChooser.formatNumber(val, valueBase); 
            fireTableCellUpdated(row, col);
            return;
         }
      
      
        /*
         * Update cell contents in table model.  Does not affect MIPS memory.
         */
          private void setDisplayAndModelValueAt(Object value, int row, int col) {
            data[row][col] = value;
            fireTableCellUpdated(row, col);
         }
      
          private void printDebugData() {
            int numRows = getRowCount();
            int numCols = getColumnCount();
         
            for (int i=0; i < numRows; i++) {
               System.out.print("    row " + i + ":");
               for (int j=0; j < numCols; j++) {
                  System.out.print("  " + data[i][j]);
               }
               System.out.println();
            }
            System.out.println("--------------------------");
         }
      }  
       ///////////////////////////////////////////////////////////////////
   	 //
   	 // JTable subclass to provide custom tool tips for each of the
   	 // text table column headers. From Sun's JTable tutorial.
   	 // http://java.sun.com/docs/books/tutorial/uiswing/components/table.html
   	 //
       private class MyTippedJTable extends JTable {
          MyTippedJTable(DataTableModel m) {
            super(m);
         }       
         
         private String[] columnToolTips = {
               /* address  */ "Base MIPS memory address for this row of the table.",
               /* value +0 */ "32-bit value stored at base address for its row.",
            	/* value +n */ "32-bit value stored ",
            	/* value +n */ " bytes beyond base address for its row."
               };
         	
          //Implement table header tool tips. 
          protected JTableHeader createDefaultTableHeader() {
            return 
                new JTableHeader(columnModel) {
                   public String getToolTipText(MouseEvent e) {
                     String tip = null;
                     java.awt.Point p = e.getPoint();
                     int index = columnModel.getColumnIndexAtX(p.x);
                     int realIndex = columnModel.getColumn(index).getModelIndex();
                     return (realIndex < 2) ?  columnToolTips[realIndex]
                                        : columnToolTips[2]+((realIndex-1)*4)+columnToolTips[3];
                  }
               };
         }
      }
   }