   package mars.util;
   import mars.*;
   import java.io.*;
   import java.util.zip.*;
   import java.util.*;
   import javax.swing.filechooser.FileFilter;
	
	/*
Copyright (c) 2003-2006,  Pete Sanderson and Kenneth Vollmar

Developed by Pete Sanderson (psanderson@otterbein.edu)
and Kenneth Vollmar (kenvollmar@missouristate.edu)

Permission is hereby granted, free of charge, to any person obtaining 
a copy of this software and associated documentation files (the 
"Software"), to deal in the Software without restriction, including 
without limitation the rights to use, copy, modify, merge, publish, 
distribute, sublicense, and/or sell copies of the Software, and to 
permit persons to whom the Software is furnished to do so, subject 
to the following conditions:

The above copyright notice and this permission notice shall be 
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR 
ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF 
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

(MIT license, http://www.opensource.org/licenses/mit-license.html)
 */
	
/**
 * Utility class to perform necessary file-related search
 * operations.  One is to find file names in JAR file,
 * another is to find names of files in given directory
 * of normal file system.
 *
 * @author Pete Sanderson
 * @version October 2006
  */
    public class FilenameFinder
   {
      private static final String JAR_EXTENSION = "jar";
      private static final boolean NO_DIRECTORIES = false;
      public static String MATCH_ALL_EXTENSIONS = "*"; 
   /**
    * Locate files and return list of file names.  Given a known relative directory path,
    * it will locate it and build list of all names of files in that directory 
    * having the given file extension. If the "known file path" doesn't work 
    * because MARS is running from an executable JAR file, it will locate the 
    * directory in the JAR file and proceed from there.  NOTE: since this uses
    * the class loader to get the resource, the directory path needs to be 
    * relative to classpath, not absolute.  To work with an arbitrary file system,
    * use the other version of this overloaded method.  Will NOT match directories
    * that happen to have the desired extension.
    * @param classLoader class loader to use
    * @param directoryPath Search will be confined to this directory.  Use "/" as 
    * separator but do NOT include starting or ending "/"  (e.g. mars/tools)
    * @param fileExtension Only files with this extension will be added 
    * to the list.  Do NOT include the "." in extension.
    * @return array list of matching file names as Strings.  If none, list is empty.
    */
       public static ArrayList getFilenameList(ClassLoader classLoader,
                                              String directoryPath, 
       													 String fileExtension  ) {
         fileExtension = checkFileExtension(fileExtension);
         ArrayList filenameList = new ArrayList();
         File f = new File(classLoader.getResource(directoryPath).getPath().replaceAll("%20"," "));
         File[] files = f.listFiles();
         if (files == null) {
            if (f.toString().toLowerCase().indexOf(JAR_EXTENSION)>0) {
               // Must be running from a JAR file. Use ZipFile to find files and create list.
               filenameList = getListFromJar(extractJarFilename(f.toString()), directoryPath, fileExtension);
            } 
         } 
         else {  // have array of File objects; convert to names and add to list
            FileFilter filter = getFileFilter(fileExtension, "", NO_DIRECTORIES);
            for (int i=0; i<files.length; i++) {
               if (filter.accept(files[i])) { 
                  filenameList.add(files[i].getName());
               }
            }
         }
         return filenameList;
      }
   
   
   /**
    * Locate files and return list of file names.  Given a known relative directory path,
    * it will locate it and build list of all names of files in that directory 
    * having the given file extension. If the "known file path" doesn't work 
    * because MARS is running from an executable JAR file, it will locate the 
    * directory in the JAR file and proceed from there.  NOTE: since this uses
    * the class loader to get the resource, the directory path needs to be 
    * relative to classpath, not absolute.  To work with an arbitrary file system,
    * use the other version of this overloaded method.
    * @param classLoader class loader to use
    * @param directoryPath Search will be confined to this directory.  Use "/" as 
    * separator but do NOT include starting or ending "/"  (e.g. mars/tools)
    * @param fileExtensions ArrayList of Strings containing file extensions.
    * Only files with an extension in this list will be added to the list.
    * Do NOT include the ".", eg "class" not ".class".  If Arraylist or 
    * extension null or empty, all files are added.
    * @return array list of matching file names as Strings.  If none, list is empty.
    */
       public static ArrayList getFilenameList(ClassLoader classLoader,
                                              String directoryPath, 
       													 ArrayList fileExtensions  ) {
         ArrayList filenameList = new ArrayList();
         String fileExtension;
         if (fileExtensions==null || fileExtensions.size()==0) {
            filenameList = getFilenameList(classLoader,directoryPath,"");
         } 
         else {
            for (int i=0; i<fileExtensions.size(); i++) {
               fileExtension = checkFileExtension((String)fileExtensions.get(i));
               filenameList.addAll(getFilenameList(classLoader,directoryPath, fileExtension));
            }
         }
         return filenameList;
      }
   
   
   /**
    * Locate files and return list of file names.  Given a known directory path,
    * it will locate it and build list of all names of files in that directory 
    * having the given file extension.  If file extenion is null or empty, all
    * filenames are returned. Returned list contains absolute filename paths.
    * @param directoryPath Search will be confined to this directory.  
    * @param fileExtension Only files with this extension will be added to the list.
    * Do NOT include "." in extension.
    * If null or empty string, all files are added.
    * @return array list of matching file names (absolute path).  If none, list is empty. 
    */
       public static ArrayList getFilenameList(String directoryPath, String fileExtension) {
         fileExtension = checkFileExtension(fileExtension);
         ArrayList filenameList = new ArrayList();
         File directory = new File(directoryPath);
         if (directory.isDirectory()) {
            File[] allFiles = directory.listFiles();
            FileFilter filter = getFileFilter(fileExtension, "", NO_DIRECTORIES);
            for (int i=0; i<allFiles.length; i++) {
               if (filter.accept(allFiles[i])) {
                  filenameList.add(allFiles[i].getAbsolutePath());
               }
            }
         }
         return filenameList;
      }
   
   
   /**
    * Locate files and return list of file names.  Given a known directory path,
    * it will locate it and build list of all names of files in that directory 
    * having the given file extension.  If file extenion is null or empty, all
    * filenames are returned. Returned list contains absolute filename paths.
    * @param directoryPath Search will be confined to this directory.  
    * @param fileExtensions ArrayList of Strings containing file extensions.
    * Only files with an extension in this list will be added 
    * to the list.  Do NOT include the "." in extensions.  If Arraylist or 
    * extension null or empty, all files are added.
    * @return array list of matching file names (absolute path).  If none, list is empty. 
    */
       public static ArrayList getFilenameList(String directoryPath, ArrayList fileExtensions) {
         ArrayList filenameList = new ArrayList();
         String fileExtension;
         if (fileExtensions==null || fileExtensions.size()==0) {
            filenameList = getFilenameList(directoryPath,"");
         } 
         else {
            for (int i=0; i<fileExtensions.size(); i++) {
               fileExtension = checkFileExtension((String)fileExtensions.get(i));
               filenameList.addAll(getFilenameList(directoryPath, fileExtension));
            }
         }
         return filenameList;
      }
   
   
   	
   /**
    * Return list of file names.  Given a list of file names, it will return the list 
    * of all having the given file extension.  If file extenion is null or empty, all
    * filenames are returned.  Returned list contains absolute filename paths.
    * @param nameList ArrayList of String containing file names.  
    * @param fileExtension Only files with this extension will be added to the list.
    * If null or empty string, all files are added.  Do NOT include "." in extension.
    * @return array list of matching file names (absolute path).  If none, list is empty. 
    */
       public static ArrayList getFilenameList(ArrayList nameList, String fileExtension) {
         fileExtension = checkFileExtension(fileExtension);
         ArrayList filenameList = new ArrayList();
         FileFilter filter = getFileFilter(fileExtension, "", NO_DIRECTORIES);
         for (int i=0; i<nameList.size(); i++) {
            File file = new File((String)nameList.get(i));
            if (filter.accept(file)) {
               filenameList.add(file.getAbsolutePath());
            }				
         }
         return filenameList;
      }
   
   
   /**
    * Return list of file names.  Given a list of file names, it will return the list 
    * of all having the given file extension.  If file extenion is null or empty, all
    * filenames are returned.  Returned list contains absolute filename paths.
    * @param nameList ArrayList of String containing file names.  
    * @param fileExtensions ArrayList of Strings containing file extensions.
    * Only files with an extension in this list will be added 
    * to the list.  Do NOT include the "." in extensions.  If Arraylist or 
    * extension null or empty, all files are added.
    * @return array list of matching file names (absolute path).  If none, list is empty. 
    */
       public static ArrayList getFilenameList(ArrayList nameList, ArrayList fileExtensions) {
         ArrayList filenameList = new ArrayList();
         String fileExtension;
         if (fileExtensions==null || fileExtensions.size()==0) {
            filenameList = getFilenameList(nameList,"");
         } 
         else {
            for (int i=0; i<fileExtensions.size(); i++) {
               fileExtension = checkFileExtension((String)fileExtensions.get(i));
               filenameList.addAll(getFilenameList(nameList, fileExtension));
            }
         }
         return filenameList;
      }
   
   	/**
   	 *  Get the filename extension of the specified File.
   	 *  @param file the File object representing the file of interest
   	 *  @return The filename extension (everything that follows 
   	 *  last '.' in filename) or null if none.
   	 */
   	 // Source code from Sun Microsystems "The Java Tutorials : How To Use File Choosers"
       public static String getExtension(File file) {
         String ext = null; 
         String s = file.getName(); 
         int i = s.lastIndexOf('.'); 
         if (i > 0 && i < s.length() - 1) { 
            ext = s.substring(i+1).toLowerCase(); 
         } 
         return ext; 
      }
   	
   	/**
   	 *  Get a FileFilter that will filter files based on the given list of filename extensions.
   	 *  @param extensions ArrayList of Strings, each string is acceptable filename extension.
   	 *  @param description String containing description to be added in parentheses after list of extensions.
   	 *  @param acceptDirectories boolean value true if directories are accepted by the filter, false otherwise.
   	 *  @return a FileFilter object that accepts files with given extensions, and directories if so indicated.
   	 */
   	 
       public static FileFilter getFileFilter(ArrayList extensions, String description, boolean acceptDirectories) {
         return new MarsFileFilter(extensions, description, acceptDirectories);
      }
   	 
   	/**
   	 *  Get a FileFilter that will filter files based on the given list of filename extensions.
   	 *  All directories are accepted by the filter.
   	 *  @param extensions ArrayList of Strings, each string is acceptable filename extension
   	 *  @param description String containing description to be added in parentheses after list of extensions.
   	 *  @return a FileFilter object that accepts files with given extensions, and directories if so indicated.
   	 */		 
   	 
       public static FileFilter getFileFilter(ArrayList extensions, String description) {
         return getFileFilter(extensions, description, true);
      }
   	 
   	/**
   	 *  Get a FileFilter that will filter files based on the given list of filename extensions.
   	 *  @param extension String containing acceptable filename extension.
   	 *  @param description String containing description to be added in parentheses after list of extensions.
   	 *  @param acceptDirectories boolean value true if directories are accepted by the filter, false otherwise.
   	 *  @return a FileFilter object that accepts files with given extensions, and directories if so indicated.
   	 */
   	 
       public static FileFilter getFileFilter(String extension, String description, boolean acceptDirectories) {
         ArrayList extensions = new ArrayList();
         extensions.add(extension);
         return new MarsFileFilter(extensions, description, acceptDirectories);
      }
   	 
   	/**
   	 *  Get a FileFilter that will filter files based on the given list of filename extensions.
   	 *  All directories are accepted by the filter.
   	 *  @param extension String containing acceptable filename extension
   	 *  @param description String containing description to be added in parentheses after list of extensions.
   	 *  @return a FileFilter object that accepts files with given extensions, and directories if so indicated.
   	 */		 
   	 
       public static FileFilter getFileFilter(String extension, String description) {
         ArrayList extensions = new ArrayList();
         extensions.add(extension);
         return getFileFilter(extensions, description, true);
      }
   	 
   
   	 
   	 // return list of file names in specified folder inside JAR
       private static ArrayList getListFromJar(String jarName, String directoryPath, String fileExtension) {
         fileExtension = checkFileExtension(fileExtension);
         ArrayList nameList = new ArrayList();
         if (jarName==null) {
            return nameList;
         }
         try {
            ZipFile zf = new ZipFile(new File(jarName));
            Enumeration list = zf.entries();
            while (list.hasMoreElements()) {
               ZipEntry ze = (ZipEntry) list.nextElement();
               if (ze.getName().startsWith(directoryPath+"/") && 
                   fileExtensionMatch(ze.getName(),fileExtension)) {
                  nameList.add(ze.getName().substring(ze.getName().lastIndexOf('/')+1));
               }
            }
         } 
             catch (Exception e) {
               System.out.println("Exception occurred reading MarsTool list from JAR: "+e);
            }
         return nameList;
      }
      
   	 // given pathname, extract and return JAR file name (must be only element ending in ".jar")
       private static String extractJarFilename(String path) {
         StringTokenizer findTheJar = new StringTokenizer(path,"\\/");
         String pathNameElement="";
         while (findTheJar.hasMoreTokens()) {
            String pathElement = findTheJar.nextToken();
            int location = pathElement.toLowerCase().indexOf(JAR_EXTENSION);
            if (location > 0) {
               pathNameElement = pathElement.substring(0,location+JAR_EXTENSION.length());
               break;
            }
         }
         return pathNameElement;
      }
   	
   	// make sure file extension, if it is real, does not start with '.' -- remove it.
       private static String checkFileExtension(String fileExtension) {
         return (fileExtension==null || fileExtension.length()==0 || !fileExtension.startsWith("."))
                ? fileExtension
            	 : fileExtension.substring(1);
      }
   	
   	 // For assured results, extension starts with "."	(will add it if not there) 
       private static boolean fileExtensionMatch(String name, String extension) {
         return (extension==null || extension.length()==0 || name.endsWith(((extension.startsWith("."))? "" : ".")+extension));
      } 
   	
   	
   	///////////////////////////////////////////////////////////////////////////
   	//  FileFilter subclass to be instantiated by the getFileFilter method above.
   	//  This extends javax.swing.filechooser.FileFilter
   	
       private static class MarsFileFilter extends FileFilter {
      
         private ArrayList extensions;
         private String fullDescription;
         private boolean acceptDirectories;
        
          private MarsFileFilter(ArrayList extensions, String description, boolean acceptDirectories) {
            this.extensions = extensions;
            this.fullDescription = buildFullDescription(description);
            this.acceptDirectories = acceptDirectories;
         }
      	
      	// User provides descriptive phrase to be parenthesized.
      	// We will attach it to description of the extensions.  For example, if the extensions
      	// given are s and asm and the description is "assembler programs" the full description
      	// generated here will be "*.s, *.asm (assembler programs)"
          private String buildFullDescription(String description) {
            String result = description;
            if (this.extensions.size() > 0) {
               result += "  (";
            }
            for (int i=0; i<this.extensions.size(); i++) {
               String extension = (String) this.extensions.get(i);
               result += ((i==0)?"":"; ")+"*"+((extension.charAt(0)=='.')? "" : ".")+extension;
            }
            if (this.extensions.size() > 0) {
               result += ")";
            }
            return result;
         }
      	
      	// required by the abstract superclass
          public String getDescription() {
            return this.fullDescription;
         }
      	
      	// required by the abstract superclass.
          public boolean accept(File file) {
            if (file.isDirectory()) { 
               return acceptDirectories; 
            } 
            String fileExtension = getExtension(file); 
            if (fileExtension != null) { 
               for (int i=0; i<extensions.size(); i++) {
                  String extension = checkFileExtension((String)extensions.get(i));
                  if (extension.equals(MATCH_ALL_EXTENSIONS) || 
                      fileExtension.equals(extension)) {
                     return true;
                  }	
               }
            }
            return false;
         }
      	
      } // MarsFileFilter class
   		
   } // FilenameFinder class

