   package mars;
   import mars.*;
   import java.io.*;
   import java.util.*;
   import java.util.prefs.*;

/*
Copyright (c) 2003-2006,  Pete Sanderson and Kenneth Vollmar

Developed by Pete Sanderson (psanderson@otterbein.edu)
and Kenneth Vollmar (kenvollmar@missouristate.edu)

Permission is hereby granted, free of charge, to any person obtaining 
a copy of this software and associated documentation files (the 
"Software"), to deal in the Software without restriction, including 
without limitation the rights to use, copy, modify, merge, publish, 
distribute, sublicense, and/or sell copies of the Software, and to 
permit persons to whom the Software is furnished to do so, subject 
to the following conditions:

The above copyright notice and this permission notice shall be 
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR 
ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF 
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

(MIT license, http://www.opensource.org/licenses/mit-license.html)
 */

   /**
	  *  Contains various IDE settings.  Persistent settings are maintained for the
	  *  current user and on the current machine using
	  *  Java's Preference objects.  Failing that, default setting values come from
	  *  Settings.properties file.  If both of those fail, default values come from
	  *  static arrays defined in this class.  The latter can can be modified prior to 
	  *  instantiating Settings object.
	  *
	  *  NOTE: If the Preference objects fail due to security exceptions, changes to
	  *  settings will not carry over from one MARS session to the next.
	  *   @author Pete Sanderson
	  **/
	  
    public class Settings {
    /* Properties file used to hold default settings. */
      private static String settingsFile = "Settings";  
	 /////////////////////////////  PROPERTY ARRAY INDEXES /////////////////////////////  
    /* Flag to determine whether or not program being assembled is limited to basic MIPS instructions and formats. */
      private static final int extendedAssemblerEnabled = 0;
    /* Flag to determine whether or not program being assembled is limited to using register numbers instead  of names. */
      private static final int bareMachineEnabled = 1;
    /* Flag to determine whether or not a file is immediately and automatically assembled upon opening. Handy when using externa editor like mipster. */
      private static final int assembleOnOpenEnabled = 2;
    /* Flag to determine whether only the current editor source file (enabled false) or all files in its
       directory (enabled true) will be assembled when assembly is selected. */
      private static final int assembleAllEnabled = 3;
    /* Default visibilty of label window (symbol table).  Default only, dynamic status maintained by ExecutePane */
      private static final int labelWindowVisibility = 4;
    /* Default setting for displaying addresses and values in hexidecimal in the Execute pane.*/
      private static final int displayAddressesInHex = 5;
      private static final int displayValuesInHex = 6;
    /* Flag to determine whether the currently selected exception handler source file will be included
       in each assembly operation. */
      private static final int exceptionHandlerEnabled = 7;	
   
      // NOTE: key sequence must match up with labels above which are used for array indexes!
      private static String[] booleanSettingsKeys = {"ExtendedAssembler", "BareMachine", "AssembleOnOpen", "AssembleAll",
                                             "LabelWindowVisibility", "DisplayAddressesInHex", "DisplayValuesInHex",
         												"LoadExceptionHandler"};
   
      /** Last resort default values for boolean settings; 
   	*   will use only if neither the Preferences nor the properties file work.
   	 *  If you wish to change them, do so before instantiating the Settings object.
   	 *  Values are matched to keys by list position: ExtendedAssembler, BareMachine, AssembleOnOpen,
   	 *  AssembleAll, LabelWindowVisibility, DisplayAddressesInHex, DisplayValuesInHex,
       *  LoadExceptionHandler.
   	 */
      public static boolean[] defaultBooleanSettingsValues = { // match the above list by position
                                              true, false, false, false, false, true, true, false };
   
   	
   	/* Current specified exception handler file (a MIPS assembly source file) */
      private static final int exceptionHandler = 0;
      private static String[] stringSettingsKeys = { "ExceptionHandler" };
      /** Last resort default values for String settings; 
   	 *  will use only if neither the Preferences nor the properties file work.
   	 *  If you wish to change, do so before instantiating the Settings object.
   	 *  Must match key by list position: ExceptionHandler.
   	 */
      public static String[] defaultStringSettingsValues = { "" };
      
      private boolean[] booleanSettingsValues;
      private String[] stringSettingsValues;	
      private Preferences preferences;
   
   /**
     * Create Settings object and set to saved values.  If saved values not found, will set
     * based on defaults stored in Settings.properties file.  If file problems, will set based
     * on defaults stored in this class.
     */
     
       public Settings() {
         booleanSettingsValues = new boolean[booleanSettingsKeys.length];
         stringSettingsValues = new String[stringSettingsKeys.length];
			// This determines where the values are actually stored.  Actual implementation
			// is platform-dependent.  For Windows, they are stored in Registry.  To see,
			// run regedit and browse to: HKEY_CURRENT_USER\Software\JavaSoft\Prefs\mars
		   preferences = Preferences.userNodeForPackage(this.getClass());
         initialize();
      }
   
	
   /**
     * Return whether backstepping is permitted at this time.  Backstepping is ability to undo execution
     * steps one at a time.  Available only in the IDE.  This is not a persistent setting and is not under
	  * MARS user control.
     * @return true if backstepping is permitted, false otherwise.
     */
       public boolean getBackSteppingEnabled() {
         return (Globals.program!=null && Globals.program.getBackStepper()!=null && Globals.program.getBackStepper().enabled());
      }
   
   
    /**
      * Reset settings to default values, as described in the constructor comments.
   	*/
       public void reset() {
         initialize();
      }
   	
   	
   	/**
   	 *  Setting for whether user programs limited to "bare machine" formatted basic instructions.
   	 *  This was added 8 Aug 2006 but is fixed at false for now, due to uncertainty as to what
   	 *  exactly constitutes "bare machine".
   	 *  @return true if only bare machine instructions allowed, false otherwise.
   	 */
       public boolean getBareMachineEnabled() {
         return booleanSettingsValues[bareMachineEnabled];
      }
   	  /**
   	   * Setting for whether user programs can use pseudo-instructions or extended addressing modes
   		* or alternative instruction formats (all are implemented as pseudo-instructions).
   		* @return true if pseudo-instructions and formats permitted, false otherwise.
   		*/
       public boolean getExtendedAssemblerEnabled() {
         return booleanSettingsValues[extendedAssemblerEnabled];
      }
   
   	  /**
   	   * Setting for whether selected program will be automatically assembled upon opening. This
   		* can be useful if user employs an external editor such as MIPSter.
   		* @return true if file is to be automatically assembled upon opening and false otherwise.
   		*/
       public boolean getAssembleOnOpenEnabled() {
         return booleanSettingsValues[assembleOnOpenEnabled];
      }
   
   	  /**
   	   * Setting for whether Addresses in the Execute pane will be displayed in hexadecimal.
   		* @return true if addresses are displayed in hexadecimal and false otherwise (decimal).
   		*/
       public boolean getDisplayAddressesInHex() {
         return booleanSettingsValues[displayAddressesInHex];
      }
   
   	  /**
   	   * Setting for whether values in the Execute pane will be displayed in hexadecimal.
   		* @return true if values are displayed in hexadecimal and false otherwise (decimal).
   		*/
       public boolean getDisplayValuesInHex() {
         return booleanSettingsValues[displayValuesInHex];
      }
   
   	  /**
   	   * Setting for whether the assemble operation applies only to the file currently open in
   		* the editor or whether it applies to all files in that file's directory (primitive project
   		* capability).  If the "assemble on open" setting is set, this "assemble all" setting will
   		* be applied as soon as the file is opened.
   		* @return true if all files are to be assembled, false if only the file open in editor.
   		*/
       public boolean getAssembleAllEnabled() {
         return booleanSettingsValues[assembleAllEnabled];
      }   		
   
   		   	  
   	 /**
   	   * Setting for whether the currently selected exception handler
   		* (a MIPS source file) will be automatically included in each
   		* assemble operation.
   		@return true if exception handler is to be included in assemble, false otherwise.
   		*/
       public boolean getExceptionHandlerEnabled() {
         return booleanSettingsValues[exceptionHandlerEnabled];
      }	
   
   	
   	/**
   	 * Default concerning whether or not to display the Labels Window -- symbol table.
   	 * @return true if label window is to be displayed, false otherwise.
   	 */
       public boolean getLabelWindowVisibility() {
         return booleanSettingsValues[labelWindowVisibility];
      }
   
   	
   	/**
   	 * Name of currently selected exception handler file.
   	 * @return String pathname of current exception handler file, null if none.
   	 */
       public String getExceptionHandler() {
         return stringSettingsValues[exceptionHandler];
      }
   
   	
   	/**
   	 * Establish setting for whether or not pseudo-instructions and formats are permitted
   	 * in user programs.  User can change this setting via the IDE.  If setting changes,
   	 * new setting will be written to properties file.
   	 * @param value True to permit, false otherwise.
   	 */
       public void setExtendedAssemblerEnabled(boolean value) {
         setBooleanSetting(extendedAssemblerEnabled, value);
      }
   	
   	 /**
   	   * Establish setting for whether a file will be automatically assembled as soon as it
   		* is opened.  This is handy for those using an external text editor such as Mipster.
   		* If setting changes, new setting will be written to properties file.
   
   	   * @param value True to automatically assemble, false otherwise.
   		*/
       public void setAssembleOnOpenEnabled(boolean value) {
         setBooleanSetting(assembleOnOpenEnabled, value);
      }
   		   	  
   	 /**
   	   * Establish setting for whether a file will be assembled by itself (false) or along
   		* with all other files in its directory (true).  This permits multi-file programs
   		* and a primitive "project" capability.  If setting changes,
   	   * new setting will be written to properties file.
   		* @param value True to assemble all, false otherwise.
   		*/
       public void setAssembleAllEnabled(boolean value) {
         setBooleanSetting(assembleAllEnabled, value);
      }
   
   	 /**
   	   * Establish setting for whether addresses in the Execute pane will be displayed
   		* in hexadecimal format.
   		* @param value True to display addresses in hexadecimal, false for decimal.
   		*/
       public void setDisplayAddressesInHex(boolean value) {
         setBooleanSetting(displayAddressesInHex, value);
      }
   
   	 /**
   	   * Establish setting for whether values in the Execute pane will be displayed
   		* in hexadecimal format.
   		* @param value True to display values in hexadecimal, false for decimal.
   		*/
       public void setDisplayValuesInHex(boolean value) {
         setBooleanSetting(displayValuesInHex, value);
      }
   
   	 /**
   	   * Establish setting for whether the labels window (i.e. symbol table) will
   		* be displayed as part of the Text Segment display.  If setting changes,
   	   * new setting will be written to properties file.
   		* @param value True to dispay labels window, false otherwise.
   		*/
       public void setLabelWindowVisibility(boolean value) {
         setBooleanSetting(labelWindowVisibility, value);
      }
   		   	   		   	  
   	 /**
   	   * Establish setting for whether the currently selected exception handler
   		* (a MIPS source file) will be automatically included in each
   		* assemble operation. If setting changes, new setting will be written 
   		* to properties file.
   		* @param value True to assemble exception handler, false otherwise
   		*/
       public void setExceptionHandlerEnabled(boolean value) {
         setBooleanSetting(exceptionHandlerEnabled, value);
      }					  

   	 
   	 /**
   	   * Set name of exception handler file and write it to settings properties file.
   		* @param newFilename name of exception handler file
   		*/
       public void setExceptionHandler(String newFilename) {
		    setStringSetting(exceptionHandler, newFilename);
      }
		
   
	/////////////////////////////////////////////////////////////////////////
	//
	//     PRIVATE HELPER METHODS TO DO THE REAL WORK
	//
	/////////////////////////////////////////////////////////////////////////

     // Initialize settings to default values.
     // Strategy: First set from properties file.  
     //           If that fails, set from array.
     //           In either case, use these values as defaults in call to Preferences.
   	
       private void initialize() {
         if (!readSettingsFromPropertiesFile(settingsFile)) {
            System.out.println("MARS System error: unable to read Settings.properties defaults. Using built-in defaults.");
            settingsHardwired();
         }
         getSettingsFromPreferences();	
      }
   	
   	// Last resort.  Use these if problem with settings property file.
       private void settingsHardwired() {
         for (int i=0; i<booleanSettingsValues.length; i++) {
            booleanSettingsValues[i] = defaultBooleanSettingsValues[i];
         }
         for (int i=0; i<stringSettingsValues.length; i++) {
            stringSettingsValues[i] = defaultStringSettingsValues[i];
         }
      }
		
			
      // Used by all the boolean setting "setter" methods.
       private void setBooleanSetting(int settingIndex, boolean value) {
         if (value != booleanSettingsValues[settingIndex]) {
            booleanSettingsValues[settingIndex] = value;
            saveBooleanSetting(settingIndex);
         }		 
      }

		
		// Used by setter method(s) for string-based settings (initially, only exception handler name)
		 private void setStringSetting(int settingIndex, String value) {
		    stringSettingsValues[settingIndex] = value;
			 saveStringSetting(settingIndex);
			}
   	
		
   	// Establish the settings from the given properties file.  Return true if it worked,
   	// false if it didn't.  Note the properties file exists only to provide default values
		// in case the Preferences fail or have not been recorded yet.
       private boolean readSettingsFromPropertiesFile(String filename) {
         try {
            for (int i=0; i<booleanSettingsKeys.length; i++) {
               booleanSettingsValues[i] = Boolean.valueOf(Globals.getPropertyEntry(filename, booleanSettingsKeys[i])).booleanValue();
            }
            for (int i=0; i<stringSettingsKeys.length; i++) {
               stringSettingsValues[i] = Globals.getPropertyEntry(filename, stringSettingsKeys[i]);
            }
         } 
             catch (Exception e) {
               return false;
            }
         return true;
      }
   	
		
		// Get settings values from Preferences object.  A key-value pair will only be written
		// to Preferences if/when the value is modified.  If it has not been modified, the default value
		// will be returned here.
		//
		// PRECONDITION: Values arrays have already been initialized to default values from
		// Settings.properties file or default value arrays above!
       private void getSettingsFromPreferences() {
         for (int i=0; i<booleanSettingsKeys.length; i++) {
            booleanSettingsValues[i] = preferences.getBoolean(booleanSettingsKeys[i], booleanSettingsValues[i]);
         }
         for (int i=0; i<stringSettingsKeys.length; i++) {
            stringSettingsValues[i] = preferences.get(stringSettingsKeys[i], stringSettingsValues[i]);
         }
      }
   	
		
		// Save the key-value pair in the Properties object and assure it is written to persisent storage.
       private void saveBooleanSetting(int index) {
         try {
			   preferences.putBoolean(booleanSettingsKeys[index], booleanSettingsValues[index]);
				preferences.flush();
				} catch (SecurityException se) {
				// cannot write to persistent storage for security reasons
				} catch (BackingStoreException bse) {
				// unable to communicate with persistent storage (strange days)
				}
      }


		// Save the key-value pair in the Properties object and assure it is written to persisent storage.		
		private void saveStringSetting(int index) {
         try {
			   preferences.put(stringSettingsKeys[index], stringSettingsValues[index]);
				preferences.flush();
				} catch (SecurityException se) {
				// cannot write to persistent storage for security reasons
				} catch (BackingStoreException bse) {
				// unable to communicate with persistent storage (strange days)
				}		
		}
   	  
   }