   package mars;
   import mars.assembler.*;
   import mars.mips.instructions.*;
   import mars.mips.hardware.*;
   import mars.util.*;
   import java.util.*;

/*
Copyright (c) 2003-2006,  Pete Sanderson and Kenneth Vollmar

Developed by Pete Sanderson (psanderson@otterbein.edu)
and Kenneth Vollmar (kenvollmar@missouristate.edu)

Permission is hereby granted, free of charge, to any person obtaining 
a copy of this software and associated documentation files (the 
"Software"), to deal in the Software without restriction, including 
without limitation the rights to use, copy, modify, merge, publish, 
distribute, sublicense, and/or sell copies of the Software, and to 
permit persons to whom the Software is furnished to do so, subject 
to the following conditions:

The above copyright notice and this permission notice shall be 
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR 
ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF 
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

(MIT license, http://www.opensource.org/licenses/mit-license.html)
 */

/**
 * Represents one assembly/machine statement.  This represents the "bare machine" level.
 * Pseudo-instructions have already been processed at this point and each assembly 
 * statement generated by them is one of these.
 * 
 * @author Pete Sanderson and Jason Bumgarner 
 * @version August 2003
 */


    public class ProgramStatement {
      private MIPSprogram sourceMIPSprogram;
      private String source, basicAssemblyStatement, machineStatement;
      private TokenList originalTokenList, strippedTokenList;
      private int[] operands;
      private int numOperands;
      private Instruction instruction;
      private int textAddress;
      private int sourceLine;
      private int binaryStatement;
      private boolean altered;
    
    //////////////////////////////////////////////////////////////////////////////////
    /**
     * Constructor for ProgramStatement when there are links back to all source and token 
     * information.  These can be used by a debugger later on.
     * @param sourceMIPSprogram The MIPSprogram object that contains this statement
     * @param source The corresponding MIPS source statement.
     * @param origTokenList Complete list of Token objects (includes labels, comments, parentheses, etc)
     * @param strippedTokenList List of Token objects with all but operators and operands removed.
     * @param inst The Instruction object for this statement's operator.
     * @param textAddress The Text Segment address in memory where the binary machine code for this statement
     * is stored.
     **/
       public ProgramStatement(MIPSprogram sourceMIPSprogram, String source, TokenList origTokenList, TokenList strippedTokenList,
                            Instruction inst, int textAddress, int sourceLine) {
         this.sourceMIPSprogram = sourceMIPSprogram;
         this.source = source;
         this.originalTokenList = origTokenList;
         this.strippedTokenList = strippedTokenList;
         this.operands = new int[4];
         this.numOperands = 0;
         this.instruction = inst;
         this.textAddress = textAddress;
         this.sourceLine = sourceLine;
         this.basicAssemblyStatement = null;
         this.machineStatement = null;
         this.binaryStatement = 0;  // nop, or sll $0, $0, 0  (32 bits of 0's)
         this.altered = false;
      }
   
   
    //////////////////////////////////////////////////////////////////////////////////
    /**
     * Constructor for ProgramStatement used only for writing a binary machine 
     * instruction with no source code to refer back to.  It could be extended to support
     * self-modifying code but currently supports only NOP which is all zeroes.
     * @param binaryStatement The 32-bit machine code.
     * @param textAddress The Text Segment address in memory where the binary machine code for this statement
     * is stored.
     **/
    //////////////////////////////////////////////////////////////////////////////////
   
       public ProgramStatement(int binaryStatement, int textAddress) {
         this.sourceMIPSprogram = null;
         this.binaryStatement = binaryStatement;
         this.textAddress = textAddress;
         this.originalTokenList = this.strippedTokenList = null;
         this.source = "";
         this.machineStatement = this.basicAssemblyStatement = null;
         this.operands = null;
         this.numOperands = 0;
         this.instruction = (binaryStatement==0) // this is a "nop" statement
                            ? (Instruction) Globals.instructionSet.matchOperator("nop").get(0)
            					 : null;
         this.altered = false;
      }
   
   
    /////////////////////////////////////////////////////////////////////////////
    /**
     * Given specification of BasicInstruction for this operator, build the
     * corresponding assembly statement in basic assembly format (e.g. substituting
     * register numbers for register names, replacing labels by values).
     * @param errors The list of assembly errors encountered so far.  May add to it here.
     **/
       public void buildBasicStatementFromBasicInstruction(ErrorList errors) {
         Token token = strippedTokenList.get(0);
         String basic = token.getValue()+" "; // the operator
         TokenTypes tokenType, nextTokenType;
         String tokenValue;
         int registerNumber;
         this.numOperands = 0;
         for (int i=1; i<strippedTokenList.size(); i++) {
            token = strippedTokenList.get(i);
            tokenType = token.getType();
            tokenValue = token.getValue();
            if (tokenType == TokenTypes.REGISTER_NUMBER) {
               basic = basic + tokenValue;
               try {
                  registerNumber = RegisterFile.getUserRegister(tokenValue).getNumber();
               } 
                   catch (Exception e) {
                    // should never happen; should be caught before now...
                     errors.add(new ErrorMessage(this.sourceMIPSprogram, token.getSourceLine(), token.getStartPos(),"invalid register name"));
                     return;
                  }
               this.operands[this.numOperands++] = registerNumber;
            } 
            else if (tokenType == TokenTypes.REGISTER_NAME) {
               registerNumber = RegisterFile.getNumber(tokenValue);
               basic = basic + "$" + registerNumber;
               if (registerNumber < 0) {
                    // should never happen; should be caught before now...
                  errors.add(new ErrorMessage(this.sourceMIPSprogram, token.getSourceLine(), token.getStartPos(),"invalid register name"));
                  return;
               }
               this.operands[this.numOperands++] = registerNumber;
            } 
            else if (tokenType == TokenTypes.FP_REGISTER_NAME) {
               registerNumber = Coprocessor1.getRegisterNumber(tokenValue);
               basic = basic + "$f" + registerNumber;
               if (registerNumber < 0) {
                    // should never happen; should be caught before now...
                  errors.add(new ErrorMessage(this.sourceMIPSprogram, token.getSourceLine(), token.getStartPos(),"invalid FPU register name"));
                  return;
               }
               this.operands[this.numOperands++] = registerNumber;
            } 
            else if (tokenType == TokenTypes.IDENTIFIER) {
               int address = this.sourceMIPSprogram.getLocalSymbolTable().getAddressLocalOrGlobal(tokenValue);
               if (address == SymbolTable.NOT_FOUND) { // symbol used without being defined
                  errors.add(new ErrorMessage(this.sourceMIPSprogram, token.getSourceLine(), token.getStartPos(),
                                   "Symbol \""+tokenValue+"\" not found in symbol table."));
                  return;
               }
            	 //////////////////////////////////////////////////////////////////////
            	 // added code 12-20-2004. If basic instruction with I_BRANCH format, then translate
            	 // address from absolute to relative and shift left 2. 
               if (instruction instanceof BasicInstruction) {
                  BasicInstructionFormat format = ((BasicInstruction)instruction).getInstructionFormat();
                  if (format ==  BasicInstructionFormat.I_BRANCH_FORMAT) {
                     address = (address - this.textAddress) >> 2;
                  }
               }
            	 //////////////////////////////////////////////////////////////////////
               basic = basic + address;
               this.operands[this.numOperands++] = address;
            } 
            else if (tokenType == TokenTypes.INTEGER_5 || tokenType == TokenTypes.INTEGER_16 ||
                       tokenType == TokenTypes.INTEGER_32) {
                // basic += tokenValue; // record as original source string
            
                ///// Begin modification 1/7/05 KENV   ///////////////////////////////////////////
            	 // We have decided to interpret non-signed (no + or -) 16-bit hexadecimal immediate  
            	 // operands as signed values in the range -32768 to 32767. So 0xffff will represent
            	 // -1, not 65535 (bit 15 as sign bit), 0x8000 will represent -32768 not 32768.
            	 // NOTE: 32-bit hexadecimal immediate operands whose values fall into this range
            	 // will be likewise affected, but they are used only in pseudo-instructions.  The
            	 // code in ExtendedInstruction.java to split this number into upper 16 bits for "lui" 
            	 // and lower 16 bits for "ori" works with the original source code token, so it is 
            	 // not affected by this tweak.  32-bit immediates in data segment directives
            	 // are also processed elsewhere so are not affected either.
            	 ////////////////////////////////////////////////////////////////////////////////
               int tempNumeric = Binary.stringToInt(tokenValue);
               if ( Binary.isHex(tokenValue) &&
                    (tempNumeric >= 32768) &&
                    (tempNumeric <= 65535) )  // Range 0x8000 ... 0xffff
               {
                     // Subtract the 0xffff bias, because strings in the
                     // range "0x8000" ... "0xffff" are used to represent
                     // 16-bit negative numbers, not positive numbers.
                  tempNumeric = tempNumeric - 65536;
                     // Note: no action needed for range 0xffff8000 ... 0xffffffff
               }
               basic += tempNumeric;  // record as decimal not original source string PS 7/20/06
               this.operands[this.numOperands++] = tempNumeric;
                ///// End modification 1/7/05 KENV   ///////////////////////////////////////////
            
            } 
            else {
               basic = basic + tokenValue;
            }
            // add separator if not at end of token list AND neither current nor 
            // next token is a parenthesis
            if ((i < strippedTokenList.size()-1)) {
               nextTokenType = strippedTokenList.get(i+1).getType();
               if (tokenType != TokenTypes.LEFT_PAREN  &&  tokenType != TokenTypes.RIGHT_PAREN  &&
                   nextTokenType != TokenTypes.LEFT_PAREN && nextTokenType != TokenTypes.RIGHT_PAREN)
                  basic = basic + ",";
            }
         }
         this.basicAssemblyStatement = basic;
      } //buildBasicStatementFromBasicInstruction()
    
    
   
    /////////////////////////////////////////////////////////////////////////////
    /**
     * Given the current statement in Basic Assembly format (see above), build the
     * 32-bit binary machine code statement.
     * @param errors The list of assembly errors encountered so far.  May add to it here.
     **/
       public void buildMachineStatementFromBasicStatement(ErrorList errors) {
         this.machineStatement = ((BasicInstruction)instruction).getOperationMask();
        //mask indicates bit positions for 'f'irst, 's'econd, 't'hird operand
         BasicInstructionFormat format = ((BasicInstruction)instruction).getInstructionFormat();
         if (format == BasicInstructionFormat.J_FORMAT) {
            // Note the  bit shift to make this a word address.
            this.operands[0] = this.operands[0] >> 2;
            this.insertBinaryCode(this.operands[0], Instruction.operandMask[0], errors); 
            if ((this.textAddress & 0xFF000000) != (this.operands[0] & 0xFF000000)) {
               // attempt to jump beyond 28-bit byte (26-bit word) address range. 
					// SPIM flags as warning, I'll flag as error b/c MARS text segment not long enough for it to be OK.
               errors.add(new ErrorMessage(this.sourceMIPSprogram, this.sourceLine, 0,"Jump target word address beyond 26-bit range"));
               return;
            }
         } 
         else if (format == BasicInstructionFormat.I_BRANCH_FORMAT) { 
            for (int i=0; i<this.numOperands-1; i++) {
               this.insertBinaryCode(this.operands[i], Instruction.operandMask[i], errors);
            }
            this.insertBinaryCode(operands[this.numOperands-1], Instruction.operandMask[this.numOperands-1], errors);         
         } 
         else {  // R_FORMAT or I_FORMAT
            for (int i=0; i<this.numOperands; i++)
               this.insertBinaryCode(this.operands[i], Instruction.operandMask[i], errors);
         }
         this.binaryStatement = Binary.binaryStringToInt(this.machineStatement);
         return;
      } // buildMachineStatementFromBasicStatement(
        
    
    //////////////////////////////////////////////////////////////////////////////
    //  Given operand (register or integer) and mask character ('f', 's', or 't'),
    //  generate the correct sequence of bits and replace the mask with them.
       private void insertBinaryCode(int value, char mask, ErrorList errors) {
         int startPos = this.machineStatement.indexOf(mask);
         int endPos = this.machineStatement.lastIndexOf(mask);
         if (startPos == -1 || endPos == -1) { // should NEVER occur
            errors.add(new ErrorMessage("",0,0,"INTERNAL ERROR: mismatch in number of operands in statement vs mask"));
            return;
         }
         String bitString = Binary.intToBinaryString(value, endPos-startPos+1);
         String state = this.machineStatement.substring(0, startPos) + bitString;
         if (endPos < this.machineStatement.length()-1)
            state = state + this.machineStatement.substring(endPos+1);
         this.machineStatement = state;
         return;
      } // insertBinaryCode()
    
    
    /////////////////////////////////////////////////////////////////////////////
    /**
     * Crude attempt at building String representation of this complex structure.
     * @return A String representing the ProgramStatement.
     **/
     
       public String toString() {
        // a crude attempt at string formatting.  Where's C when you need it?
         String blanks = "                               ";
         int firstSpace = this.basicAssemblyStatement.indexOf(' ');
         String result = "["+this.textAddress+"]";
         result += blanks.substring(0, 16-result.length()) + this.basicAssemblyStatement.substring(0,firstSpace);
         result += blanks.substring(0, 24-result.length()) + this.basicAssemblyStatement.substring(firstSpace+1);;
         result += blanks.substring(0, 40-result.length()) + ";  "; // this.source;
         for (int i=0; i<this.numOperands; i++) 
            result += operands[i] + " ";
         result += "["+Binary.binaryStringToHexString(this.machineStatement)+"]";
         result += "  "+this.machineStatement.substring(0,6)+"|" + this.machineStatement.substring(6,11)+"|"+
               this.machineStatement.substring(11,16)+"|" + this.machineStatement.substring(16,21)+"|"+
               this.machineStatement.substring(21,26)+"|" + this.machineStatement.substring(26,32);
         return result;
      } // toString()
   
    /**
     * Assigns given String to be Basic Assembly statement equivalent to this source line.
     * @param statement A String containing equivalent Basic Assembly statement.
     **/
     
       public void setBasicAssemblyStatement(String statement) {
         basicAssemblyStatement = statement;
      }
   
    /**
     * Assigns given String to be binary machine code (32 characters, all of them 0 or 1)
     * equivalent to this source line.
     * @param statement A String containing equivalent machine code.
     **/
         
       public void setMachineStatement(String statement) {
         machineStatement = statement;
      }
   
    /**
     * Assigns given int to be binary machine code equivalent to this source line.
     * @param binaryCode An int containing equivalent binary machine code.
     **/
     
       public void setBinaryStatement(int binaryCode) {
         binaryStatement = binaryCode;
      }
   
   
    /**
     * associates MIPS source statement.  Used by assembler when generating basic 
     * statements during macro expansion of extended statement.
     * @param src a MIPS source statement.
     **/
         
       public void setSource(String src) {
         source = src;
      }
   
    
    /**
     * Produces MIPSprogram object representing the source file containing this statement.
     * @return The MIPSprogram object.  May be null...
     **/
       public MIPSprogram getSourceMIPSprogram() {
         return sourceMIPSprogram;
      }	
     
    /**
     * Produces String name of the source file containing this statement.
     * @return The file name.
     **/         
       public String getSourceFile() {
         return (sourceMIPSprogram == null) ? "" : sourceMIPSprogram.getFilename();
      }	
   
   
   
    /**
     * Produces MIPS source statement.
     * @return The MIPS source statement.
     **/
         
       public String getSource() {
         return source;
      }
   
    /**
     * Produces line number of MIPS source statement.
     * @return The MIPS source statement line number.
     **/
         
       public int getSourceLine() {
         return sourceLine;
      }
    
    /**
     * Produces Basic Assembly statement for this MIPS source statement.
     * @return The Basic Assembly statement.
     **/
     
       public String getBasicAssemblyStatement() {
         return basicAssemblyStatement;
      }
   
    /**
     * Produces binary machine statement as 32 character string, all '0' and '1' chars.
     * @return The String version of 32-bit binary machine code.
     **/
     
       public String getMachineStatement() {
         return machineStatement;
      }
    
    /**
     * Produces 32-bit binary machine statement as int.
     * @return The int version of 32-bit binary machine code.
     **/
       public int getBinaryStatement() {
         return binaryStatement;
      }
    /**
     * Produces token list generated from original source statement.
     * @return The TokenList of Token objects generated from original source.
     **/
       public TokenList getOriginalTokenList() {
         return originalTokenList;
      }
    /**
     * Produces token list stripped of all but operator and operand tokens.
     * @return The TokenList of Token objects generated by stripping original list of all
     * except operator and operand tokens.
     **/
       public TokenList getStrippedTokenList() {
         return strippedTokenList;
      }
    /**
     * Produces Instruction object corresponding to this statement's operator.
     * @return The Instruction that matches the operator used in this statement.
     **/
       public Instruction getInstruction() {
         return instruction;
      }
    /**
     * Produces Text Segment address where the binary machine statement is stored.
     * @return address in Text Segment of this binary machine statement.
     **/
       public int getAddress() {
         return textAddress;
      }
    /**
     * Produces int array of operand values for this statement.
     * @return int array of operand values (if any) required by this statement's operator.
     **/    
       public int[] getOperands() {
         return operands;
      }
    /**
     * Produces operand value from given array position (first operand is position 0).
     * 
     * @param i Operand position in array (first operand is position 0).
     * @return Operand value at given operand array position.  If < 0 or >= numOperands, it returns -1.
     **/   
       public int getOperand(int i) {
         if (i >= 0 && i < this.numOperands) {
            return operands[i];
         } 
         else {
            return -1;
         }
      }
   
   
   }