   package mars;
   import mars.venus.*;
   import mars.util.*;
   import mars.mips.hardware.*;
   import java.io.*;
   import java.util.*;
   import java.awt.*;
   import javax.swing.*;
   import javax.swing.JOptionPane;   // KENV 9/8/2004

/*
Copyright (c) 2003-2006,  Pete Sanderson and Kenneth Vollmar

Developed by Pete Sanderson (psanderson@otterbein.edu)
and Kenneth Vollmar (kenvollmar@missouristate.edu)

Permission is hereby granted, free of charge, to any person obtaining 
a copy of this software and associated documentation files (the 
"Software"), to deal in the Software without restriction, including 
without limitation the rights to use, copy, modify, merge, publish, 
distribute, sublicense, and/or sell copies of the Software, and to 
permit persons to whom the Software is furnished to do so, subject 
to the following conditions:

The above copyright notice and this permission notice shall be 
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR 
ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF 
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

(MIT license, http://www.opensource.org/licenses/mit-license.html)
 */

/**
 * Launch the Mars application
 * 
 * @author Pete Sanderson
 * @version August 2003
 **/

    public class MarsLaunch {
   
   /**
    * Main takes a number of command line arguments.<br>
      Usage:  Mars  [options] filename<br>
      Valid options (not case sensitive, separate by spaces) are:<br>
            a  -- assemble only, do not simulate<br>
            d  -- print debugging statements<br>
           ad  -- both a and d<br>
           da  -- both a and d<br>
          <n>  -- where <n> is an integer maximum count of steps to simulate.<br>
                  If 0, negative or not specified, there is no maximum.<br>
       $<reg>  -- where <reg> is number or name (e.g. 5, t3, f10) of register whose <br>
                  content to display at end of run.  Option may be repeated.<br>
      <m>-<n>  -- memory address range from <m> to <n> whose contents to<br>
                  display at end of run. <m> and <n> may be hex or decimal,<br>
                  <m> <= <n>, both must be on word boundary.  Option may be repeated.<br>
          hex  -- display memory or register contents in hexidecimal (default)<br>
          dec  -- display memory or register contents in decimal.<br>
   		   b  -- brief - do not display register/memory address along with contents<br>
   		  np  -- No Pseudo-instructions allowed ("ne" will work also).
   		   p  -- Project mode - assemble all files in the same directory as given file.
            h  -- display help.  Use by itself and with no filename</br>
    **/
    
   
   
      private boolean simulate;
      private boolean hexDisplay;
      private boolean verbose;  // display register name or address along with contents
      private boolean assembleProject; // assemble only the given file or all files in its directory
      private boolean pseudo;  // pseudo instructions allowed in source code or not.
      private static final String rangeSeparator = "-";
      private static final int splashDuration = 2000; // time in MS to show splash screen
      private static final int memoryWordsPerLine = 4; // display 4 memory words, tab separated, per line
      private ArrayList registerDisplayList;
      private ArrayList memoryDisplayList;
      private ArrayList filenameList;
      private MIPSprogram code;
      private int maxSteps;
      private PrintStream out; // stream for display of command line output
   		
       public MarsLaunch(String[] args) {
         Globals.initialize();  
         if (args.length == 0) {
            launchIDE();
         } 
         else {
            simulate = true;
            hexDisplay = true;
            verbose = true;  
            assembleProject = false;
            pseudo = true;
            registerDisplayList = new ArrayList();
            memoryDisplayList = new ArrayList();
            filenameList = new ArrayList();
         	// do NOT use Globals.program for command line MARS -- it triggers 'backstep' log.
            code = new MIPSprogram();  
            maxSteps = -1;
            out = System.out;
            if (parseCommandArgs(args)) {
               if (runCommand()) {
                  displayRegistersPostMortem();
                  displayMemoryPostMortem();
               }
            }
            System.exit(Globals.exitCode);
         }
      }   		
   	
   		
   	/////////////////////////////////////////////////////////////////
   	// There are no command arguments, so run in interactive mode by
   	// launching the GUI-fronted integrated development environment.
   	
       private void launchIDE() {
         new MarsSplashScreen(splashDuration).showSplash();
         VenusUI frame = new VenusUI("MARS "+Globals.version);
         return;					
      }
   			
   			
   	//////////////////////////////////////////////////////////////////////
   	// Parse command line arguments.  The initial parsing has already been
   	// done, since each space-separated argument is already in a String array
   	// element.  Here, we check for validity, set switch variables as appropriate
   	// and build data structures.  For help option (h), display the help.
   	// Returns true if command args parse OK, false otherwise.
      			
       private boolean parseCommandArgs(String[] args) {
         boolean argsOK = true;
         if (args.length == 0) 
            return true; // should not get here...
         out.println("MARS "+Globals.version+"  Copyright "+Globals.copyrightYears+" "+Globals.copyrightHolders+"\n");
         if (args.length == 1 && args[0].equals("h")) {
            displayHelp();
            return false;					
         } 
         for (int i=0; i<args.length; i++) {
            if (args[i].toLowerCase().equals("d")) { 
               Globals.debug = true;
               continue;
            }
            if (args[i].toLowerCase().equals("a")) { 
               simulate = false;
               continue;
            }
            if (args[i].toLowerCase().equals("ad") || 
               	 args[i].toLowerCase().equals("da")) {
               Globals.debug = true;
               simulate = false;
               continue;
            }
            if (args[i].toLowerCase().equals("p")) { 
               assembleProject = true;
               continue;
            }
            if (args[i].toLowerCase().equals("dec")) {
               hexDisplay = false;
               continue;
            }
            if (args[i].toLowerCase().equals("hex")) {
               hexDisplay = true;
               continue;
            }
            if (args[i].toLowerCase().equals("b")) {
               verbose = false;
               continue;
            }
            if (args[i].toLowerCase().equals("np") || args[i].toLowerCase().equals("ne")) {
               pseudo = false;
               continue;
            }
            if (args[i].indexOf("$") == 0) {
               if (RegisterFile.getUserRegister(args[i])==null &&
                      Coprocessor1.getRegister(args[i])==null) {
                  out.println("Invalid Register Name: "+args[i]);
               } 
               else {
                  registerDisplayList.add(args[i]);
               }
               continue;
            }
            if (new File(args[i]).exists()) {  // is it a file name?
               filenameList.add(args[i]);
               continue;
            }
            	// Check for stand-alone integer, which is the max execution steps option
            try {   
               Integer.decode(args[i]);
               maxSteps = Integer.decode(args[i]).intValue(); // if we got here, it has to be OK 
               continue;              
            } 
                catch (NumberFormatException nfe) {
               }
            	// Check for memory address subrange.  Has to be two integers separated
            	// by "-"; no embedded spaces.  e.g. 0x00400000-0x00400010
            	// If number is not multiple of 4, will be rounded up to next higher.
            if (args[i].indexOf(rangeSeparator) > 0 && 
                   args[i].indexOf(rangeSeparator) < args[i].length()-1) {
               // assume correct format, two numbers separated by -, no embedded spaces.
               // If that doesn't work it is invalid.
               String memoryRangeBegin = args[i].substring(0,args[i].indexOf(rangeSeparator));
               String memoryRangeEnd = args[i].substring(args[i].indexOf(rangeSeparator)+1);
               	// NOTE: I will use homegrown decoder, because Integer.decode will throw
               	// exception on address higher than 0x7FFFFFFF (e.g. sign bit is 1).
               try {
                  if (Binary.stringToInt(memoryRangeBegin) > Binary.stringToInt(memoryRangeEnd) ||
                         !Memory.wordAligned(Binary.stringToInt(memoryRangeBegin)) ||
                     	 !Memory.wordAligned(Binary.stringToInt(memoryRangeEnd))) {
                     throw new NumberFormatException();
                  }
                    // if we're still here, they both are valid numbers
                  memoryDisplayList.add(memoryRangeBegin);
                  memoryDisplayList.add(memoryRangeEnd);
               }
                   catch (NumberFormatException nfe) {
                     out.println("Invalid/unaligned address or invalid range: "+args[i]);
                     argsOK = false;
                  }
               continue;
            }
            out.println("Invalid Command Argument: "+args[i]);
            argsOK = false;
         }
         return argsOK;
      }
      
   	
   	//////////////////////////////////////////////////////////////////////
   	// Carry out the mars command: assemble then optionally run     
   	// Returns false if no simulation (run) occurs, true otherwise. 	
      	
       private boolean runCommand() {
         boolean programRan = false;
         if (filenameList.size()==0) {
            return programRan;
         }
         try {
            File mainFile = new File((String) filenameList.get(0));// First file is "main" file
            ArrayList filesToAssemble;
            if (assembleProject) { 
               filesToAssemble = FilenameFinder.getFilenameList(mainFile.getParent(), Globals.fileExtensions);
               if (filenameList.size() > 1) {
                  // Using "p" project option PLUS listing more than one filename on command line.
                  // Add the additional files, avoiding duplicates.
                  filenameList.remove(0); // first one has already been processed
                  ArrayList moreFilesToAssemble = FilenameFinder.getFilenameList(filenameList, FilenameFinder.MATCH_ALL_EXTENSIONS);
                  // Remove any duplicates then merge the two lists.
                  for (int index2 = 0; index2<moreFilesToAssemble.size(); index2++) {
                     for (int index1 = 0; index1<filesToAssemble.size(); index1++) {
                        if (filesToAssemble.get(index1).equals(moreFilesToAssemble.get(index2))) {
                           moreFilesToAssemble.remove(index2);
                           index2--; // adjust for left shift in moreFilesToAssemble...
                           break;    // break out of inner loop...
                        }
                     }
                  }
                  filesToAssemble.addAll(moreFilesToAssemble);
               }
            } 
            else {
               filesToAssemble = FilenameFinder.getFilenameList(filenameList, FilenameFinder.MATCH_ALL_EXTENSIONS);
            }
            if (Globals.debug) {
               out.println("--------  TOKENIZING BEGINS  -----------");
            }
            ArrayList MIPSprogramsToAssemble = 
                      code.prepareFilesForAssembly(filesToAssemble, mainFile.getAbsolutePath(), null);		
            if (Globals.debug) {
               out.println("--------  ASSEMBLY BEGINS  -----------");
            }
         	// Added logic to check for warnings and print if any. DPS 11/28/06
            ErrorList warnings = code.assemble(MIPSprogramsToAssemble, pseudo);
            if (warnings != null && warnings.warningsOccurred()) {
               out.println(warnings.generateWarningReport());
            }
            if (simulate) {
               if (Globals.debug) {
                  out.println("--------  SIMULATION BEGINS  -----------");
               }
               programRan = true;
               boolean done = code.simulate(maxSteps);
               if (!done) {
                  out.println("\nProgram terminated when maximum step limit "+maxSteps+" reached.");
               }
            }
            if (Globals.debug) {
               out.println("\n--------  ALL PROCESSING COMPLETE  -----------");
            }
         }
             catch (ProcessingException e) {
               out.println(e.errors().generateErrorAndWarningReport());
            } 
         return programRan;
      }
           
   		   	
   	//////////////////////////////////////////////////////////////////////
   	// Displays requested register or registers   			
   				
       private void displayRegistersPostMortem() {
         int value;  // handy local to use throughout the next couple loops
         // Display requested register contents
         out.println();
         Iterator regIter = registerDisplayList.iterator();
         while (regIter.hasNext()) {
            String reg = regIter.next().toString();
            if (RegisterFile.getUserRegister(reg)!=null) {
                     // integer register
               if (verbose) 
                  out.print(reg+"\t");
               value = RegisterFile.getUserRegister(reg).getValue();
               out.println( (hexDisplay) ? Binary.intToHexString(value) : ""+value);
            } 
            else {
                     // floating point register
               float fvalue = Coprocessor1.getFloatFromRegister(reg);
               double dvalue = Double.NaN;
               boolean hasDouble = false;
               try {
                  dvalue = Coprocessor1.getDoubleFromRegister(reg);
                  hasDouble = true;
               } 
                   catch (InvalidRegisterAccessException irae) { }
               if (verbose) {
                  out.print(reg+"\t");
               }
               if (hexDisplay) {
                        // display float (and double, if applicable) in hex
                  out.print(
                           Binary.binaryStringToHexString(
                                          Binary.intToBinaryString(
                           					Float.floatToRawIntBits(fvalue))));								
                  if (hasDouble) {
                     out.println("\t"+
                              Binary.binaryStringToHexString(
                                          Binary.longToBinaryString(
                              				Double.doubleToRawLongBits(dvalue))));
                  } 
                  else {
                     out.println("");
                  }
               } 
               else {
                        // display float (and double, if applicable) in decimal
                  out.print(fvalue);
                  if (hasDouble) {
                     out.println("\t"+dvalue);
                  } 
                  else {
                     out.println("");
                  }
               }
            }
         }
      }
      
   	
   	//////////////////////////////////////////////////////////////////////
   	// Displays requested memory range or ranges
   	
       private void displayMemoryPostMortem() {  
         int value;  
         // Display requested memory range contents
         Iterator memIter = memoryDisplayList.iterator();
         int addressStart=0, addressEnd=0;
         while (memIter.hasNext()) {
            try { // This will succeed; error would have been caught during command arg parse
               addressStart = Binary.stringToInt(memIter.next().toString());
               addressEnd = Binary.stringToInt(memIter.next().toString());
            } 
                catch (NumberFormatException nfe) {
               }
            int valuesDisplayed = 0;
            for (int addr=addressStart; addr<=addressEnd; addr+=Memory.WORD_LENGTH_BYTES) {
               if (valuesDisplayed % memoryWordsPerLine == 0) {
                  out.print( (valuesDisplayed > 0) ? "\n" : "");
                  if (verbose) {
                     out.print("Mem["+Binary.intToHexString(addr)+"]\t");
                  }
               }
               try {
                  value = Globals.memory.getWord(addr);
                  out.print( (hexDisplay) ? Binary.intToHexString(value)+"\t" : ""+value+"\t");
               }
                   catch (AddressErrorException aee) {
                     out.print("Invalid address: "+addr+"\t");
                  }
               valuesDisplayed++;
            }
            out.println();
         }					
      }
   
   
   	///////////////////////////////////////////////////////////////////////
   	//  Display command line help text
   	
       private void displayHelp() {
         out.println("Usage:  Mars  [options] filename [additional filenames]");
         out.println("  Valid options (not case sensitive, separate by spaces) are:");
         out.println("      a  -- assemble only, do not simulate");
         out.println("     np  -- use of pseudo instructions and formats not permitted");
         out.println("    <n>  -- where <n> is an integer maximum count of steps to simulate.");
         out.println("            If 0, negative or not specified, there is no maximum.");
         out.println(" $<reg>  -- where <reg> is number or name (e.g. 5, t3, f10) of register whose ");
         out.println("            content to display at end of run.  Option may be repeated.");
         out.println("<m>-<n>  -- memory address range from <m> to <n> whose contents to");
         out.println("            display at end of run. <m> and <n> may be hex or decimal,");
         out.println("            must be on word boundary, <m> <= <n>.  Option may be repeated.");
         out.println("    hex  -- display memory or register contents in hexadecimal (default)");
         out.println("    dec  -- display memory or register contents in decimal.");
         out.println("      b  -- brief - do not display register/memory address along with contents");
         out.println("      d  -- display MARS debugging statements");
         out.println("      p  -- Project mode - assemble all files in the same directory as given file.");
         out.println("      h  -- display this help.  Use by itself with no filename.");
         out.println("If more than one filename is listed, the first is assumed to be the main.");
         out.println("Exception handler not automatically assembled.  Add it to the file list.");
         out.println("Options used here do not affect MARS Settings menu settings and vice versa.");
      }
   
   }
   	
   	
