/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */

/*  ArchC Syscall Library for the ArchC architecture simulators
    Copyright (C) 2002-2004  The ArchC Team

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.
*/

/********************************************************/
/* ArchC Syscalls implementation file.                  */
/* Author:  Marcus Bartholomeu                          */
/*                                                      */
/*                                                      */
/* The ArchC Team                                       */
/* Computer Systems Laboratory (LSC)                    */
/* IC-UNICAMP                                           */
/* http://www.lsc.ic.unicamp.br                         */
/********************************************************/
 

#include "ac_syscall.H"
#include "archc.H"
#include "ac_resources.H"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include <sys/times.h>
#include <time.h>

#ifdef DEBUG
#  define DEBUG_SYSCALL(name) AC_RUN_MSG("@@@@@ syscall: " name " @@@@@\n")
#else
#  define DEBUG_SYSCALL(name)
#endif


// Fix incompatibility from NewLib flags and Linux flags

#define NEWLIB_O_RDONLY          0x0000
#define NEWLIB_O_WRONLY          0x0001
#define NEWLIB_O_RDWR            0x0002
#define NEWLIB_O_APPEND          0x0008
#define NEWLIB_O_CREAT           0x0200
#define NEWLIB_O_TRUNC           0x0400
#define NEWLIB_O_EXCL            0x0800
#define NEWLIB_O_NOCTTY          0x8000
#define NEWLIB_O_NONBLOCK        0x4000

#define CORRECT_O_RDONLY             00
#define CORRECT_O_WRONLY             01
#define CORRECT_O_RDWR               02
#define CORRECT_O_CREAT            0100
#define CORRECT_O_EXCL             0200
#define CORRECT_O_NOCTTY           0400
#define CORRECT_O_TRUNC           01000
#define CORRECT_O_APPEND          02000
#define CORRECT_O_NONBLOCK        04000


void correct_flags( int* val )
{
  int f = *val;
  int flags = 0;

  if( f &  NEWLIB_O_RDONLY )
    flags |= CORRECT_O_RDONLY;
  if( f &  NEWLIB_O_WRONLY )
    flags |= CORRECT_O_WRONLY;
  if( f &  NEWLIB_O_RDWR )
    flags |= CORRECT_O_RDWR;
  if( f & NEWLIB_O_CREAT )
    flags |= CORRECT_O_CREAT;
  if( f & NEWLIB_O_EXCL )
    flags |= CORRECT_O_EXCL;
  if( f & NEWLIB_O_NOCTTY )
    flags |= CORRECT_O_NOCTTY;
  if( f & NEWLIB_O_TRUNC )
    flags |= CORRECT_O_TRUNC;
  if( f & NEWLIB_O_APPEND )
    flags |= CORRECT_O_APPEND;
  if( f & NEWLIB_O_NONBLOCK )
    flags |= CORRECT_O_NONBLOCK;

  *val = flags;
}


//! Processor independent functions (syscalls)

void ac_syscall::open()
{
#ifdef AC_MEM_HIERARCHY
  if (!flush_cache()) return;
#endif
  DEBUG_SYSCALL("open");
  unsigned char pathname[100];
  get_buffer(0, pathname, 100);
  int flags = get_int(1); correct_flags(&flags);
  int mode = get_int(2);
  int ret = ::open((char*)pathname, flags, mode);
//   if (ret == -1) {
//     AC_RUN_ERROR("System Call open (file '%s'): %s\n", pathname, strerror(errno));
//     exit(EXIT_FAILURE);
//   }
  set_int(0, ret);
  return_from_syscall();
}


void ac_syscall::creat()
{
  DEBUG_SYSCALL("creat");
  unsigned char pathname[100];
  get_buffer(0, pathname, 100);
  int mode = get_int(1);
  int ret = ::creat((char*)pathname, mode);
  if (ret == -1) {
    AC_RUN_ERROR("System Call creat (file '%s'): %s\n", pathname,strerror(errno));
    exit(EXIT_FAILURE);
  }
  set_int(0, ret);
  return_from_syscall();
}


void ac_syscall::close()
{
  DEBUG_SYSCALL("close");
  int fd = get_int(0);
  int ret = ::close(fd);
  if (ret == -1) {
    AC_RUN_ERROR("System Call close (fd %d): %s\n", fd, strerror(errno));
    exit(EXIT_FAILURE);
  }
  set_int(0, ret);
  return_from_syscall();
}


void ac_syscall::read()
{
#ifdef AC_MEM_HIERARCHY
  if (!flush_cache()) return;
#endif  
  DEBUG_SYSCALL("read");
  int fd = get_int(0);
  unsigned count = get_int(2);
  unsigned char *buf = (unsigned char*) malloc(count);
  int ret = ::read(fd, buf, count);
  if (ret == -1) {
    AC_RUN_ERROR("System Call read (fd %d): %s\n", fd, strerror(errno));
    exit(EXIT_FAILURE);
  }
  set_buffer(1, buf, ret);
  set_int(0, ret);
  return_from_syscall();
  free(buf);
}


void ac_syscall::write()
{
#ifdef AC_MEM_HIERARCHY
  if (!flush_cache()) return;
#endif
  DEBUG_SYSCALL("write");
  int fd = get_int(0);
  unsigned count = get_int(2);
  unsigned char *buf = (unsigned char*) malloc(count);
  get_buffer(1, buf, count);
  int ret = ::write(fd, buf, count);
  if (ret == -1) {
    AC_RUN_ERROR("System Call write (fd %d): %s\n", fd, strerror(errno));
    exit(EXIT_FAILURE);
  }
  set_int(0, ret);
  return_from_syscall();
  free(buf);
}


void ac_syscall::isatty()
{
  DEBUG_SYSCALL("isatty");
  int desc = get_int(0);
  int ret = ::isatty(desc);
  set_int(0, ret);
  return_from_syscall();
}


void ac_syscall::sbrk()
{
  DEBUG_SYSCALL("sbrk");
  extern unsigned int ac_heap_ptr;
  unsigned int base = ac_heap_ptr;
  unsigned int increment = get_int(0);
  ac_heap_ptr += increment;

  // Test if there is enough space in the target memory 
  // OBS: 1kb is reserved at the end of memory to command line parameters
  if (ac_heap_ptr > AC_RAMSIZE-1024) {
    // Show error only once
    static bool show_error = true;
    if (show_error) {
      AC_WARN("Target application failed to allocate " << increment <<
               " bytes: heap(=" << ac_heap_ptr << ") > ramsize(=" <<
               AC_RAMSIZE << ")");
      AC_WARN("If target application does not treat allocation error, it may crash.");
    }
    show_error = false;
    ac_heap_ptr = base;
    set_int(0, -1);
  }
  else {
    set_int(0, base);
  }

  return_from_syscall();
}


void ac_syscall::lseek()
{
  DEBUG_SYSCALL("lseek");
  int fd = get_int(0);
  int offset = get_int(1);
  int whence = get_int(2);
  int ret = ::lseek(fd, offset, whence);
  set_int(0, ret);
  return_from_syscall();
}


void ac_syscall::fstat()
{
  DEBUG_SYSCALL("fstat");
  static bool fstat_warn = true;
  if (fstat_warn) {
    AC_WARN("This version of fstat should not be called!");
    AC_WARN("Please, recompile your target application with an updated libac_sysc.");
    fstat_warn = false;
  }
  int fd = get_int(0);
  struct stat buf;
  int ret = ::fstat(fd, &buf);
  if (ret == -1) {
    AC_RUN_ERROR("System Call fstat (fd %d): %s\n", fd, strerror(errno));
    exit(EXIT_FAILURE);
  }
  set_int(0, ret);
  return_from_syscall();
}


void ac_syscall::_exit()
{
  DEBUG_SYSCALL("_exit");
  int ac_exit_status = get_int(0);
#ifdef USE_GDB
  if (gdbstub) gdbstub->exit(ac_exit_status);
#endif /* USE_GDB */
  ac_stop(ac_exit_status);
}


void ac_syscall::times()
{
  DEBUG_SYSCALL("times");
  unsigned char zeros[16] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
  set_buffer(0, zeros, 16);
  set_int(0, 0);
  return_from_syscall();
}


void ac_syscall::time()
{
  DEBUG_SYSCALL("time");
  int t = get_int(0);
  int ret = ::time(0);
  if (t!=0) set_buffer(0, (unsigned char *) &ret, 4);
  set_int(0, ret);
  return_from_syscall();
}


void ac_syscall::random()
{
  DEBUG_SYSCALL("random");
  int ret = ::random();
  set_int(0, ret);
  return_from_syscall();
}


#include <ac_syscall_codes.h>

void ac_syscall::ac_syscall_wrapper()
{
  int ret = -1;
  unsigned char pathname[100];
  int mode;
  int fd, newfd;
  static struct stat buf_stat;
  //static struct tms  buf_tms;

  int syscall_code = get_int(0);

  switch(syscall_code) {

  case __NR_getpid:
    DEBUG_SYSCALL("getpid");
    ret = 123;
    break;

  case __NR_chmod:
    DEBUG_SYSCALL("chmod");
    get_buffer(0, pathname, 100);
    mode = get_int(1);
    ret = ::chmod((char*)pathname, mode);
    break;

  case __NR_dup:
    DEBUG_SYSCALL("dup");
    fd = get_int(1);
    ret = ::dup(fd);
    break;

  case __NR_dup2:
    DEBUG_SYSCALL("dup2");
    fd = get_int(1);
    newfd = get_int(2);
    ret = ::dup2(fd, newfd);
    break;

  case __NR_fstat:
    DEBUG_SYSCALL("fstat");
    fd = get_int(1);
    ret = ::fstat(fd, &buf_stat);
    break;


    /* Special cases for the fields of the "struct stat":
       to convert from glibc to newlib */

#define FILL_STRUCT_STAT(x) \
  case __AC_struct_stat_##x: \
    DEBUG_SYSCALL("filling struct stat field: " #x); \
    ret = buf_stat.x; \
    break

    FILL_STRUCT_STAT(st_dev);
    FILL_STRUCT_STAT(st_ino);
    FILL_STRUCT_STAT(st_mode);
    FILL_STRUCT_STAT(st_nlink);
    FILL_STRUCT_STAT(st_uid);
    FILL_STRUCT_STAT(st_gid);
    FILL_STRUCT_STAT(st_rdev);
    FILL_STRUCT_STAT(st_size);
    FILL_STRUCT_STAT(st_blksize);
    FILL_STRUCT_STAT(st_blocks);
    FILL_STRUCT_STAT(st_atime);
    FILL_STRUCT_STAT(st_mtime);
    FILL_STRUCT_STAT(st_ctime);

#undef FILL_STRUCT_STAT


  default:
    AC_RUN_ERROR("System Call code %d not implemented yet.\n", syscall_code);
    exit(EXIT_FAILURE);
  }
    
  //if (ret == -1)   AC_RUN_ERROR("System call %d returned -1.\n", syscall_code);
  set_int(0, ret);
  return_from_syscall();
}


void ac_syscall::ac_syscall_geterrno()
{
  set_int(0, errno);
  return_from_syscall();
}


void ac_syscall::ac_syscall_stat_mode()
{
  AC_RUN_ERROR("System Call ac_syscall_stat_mode not implemented yet.\n");
  exit(EXIT_FAILURE);
}


