/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */

/*  ArchC Storage Library for the ArchC architecture simulators
    Copyright (C) 2002-2004  The ArchC Team

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.
*/

/********************************************************/
/* The ArchC storage device base class.                 */
/* Author:  Sandro Rigo                                 */
/*                                                      */
/*                                                      */
/* The ArchC Team                                       */
/* Computer Systems Laboratory (LSC)                    */
/* IC-UNICAMP                                           */
/* http://www.lsc.ic.unicamp.br                         */
/********************************************************/

//////////////////////////////////////////////////////////
/*!\file ac_storage.cpp
  \brief The ArchC storage device base class. Implementation file.

  This class contains  basic members and methods needed by
  all storage devices for behavioral simulation. 
*/
/************************************************
 * Author: Sandro Rigo
 ************************************************/
//////////////////////////////////////////////////////////

#include <sstream>
#include <string>
#include <fstream>
#include <iterator>
#include  "ac_storage.H"
#include "ac_resources.H"
//#include "ac_types.H"

//#define DEBUG_STORAGE

//////////////////////////////////
// ac_storage methods      
//////////////////////////////////

//!Reading the content of an address.
ac_word ac_storage::read( unsigned address ) { 

#ifdef AC_STATS
  ac_resources::ac_sim_stats.add_access(name);
#endif
#ifdef DEBUG_STORAGE
  if( !Data )
    cout << "Data is uninitialized" <<endl;

  if( address > size)
    cout << "Access out of bounds" <<endl;
#endif

  return *((ac_word *)(Data+(address)));
}

//!Reading a byte.
unsigned char ac_storage::read_byte( unsigned address ) { 

  int align = sizeof(ac_word) -1;  //Indicates the memory alignment
  int byte_idx;

  ac_word word;

  //Getting the word containing the requested byte. 
  //This is necessary because the address may not be the beginning of a memory word.
  word = *((ac_word *)(Data+(address & ~align)));
  //Checking target endianess to discover which byte must be returned
  byte_idx = (ac_resources::ac_tgt_endian) ? (align - (address & align)) : (address & align); 

#ifdef AC_STATS
  ac_resources::ac_sim_stats.add_access(name);
#endif

  return  (unsigned char) ((word & (0xFF << (byte_idx *8))) >> (byte_idx*8));
}

//!Reading half word.
ac_Hword ac_storage::read_half( unsigned address ) { 

  int align = sizeof(ac_word) -1;  //Indicates the memory alignment
  int byte_idx;

  ac_word word;

  //Getting the word containing the requested byte. 
  //This is necessary because the address may not be the beginning of a memory word.
  word = *((ac_word *)(Data+(address & ~align)));
  //Checking target endianess to discover which byte must be returned
  byte_idx = (ac_resources::ac_tgt_endian) ? (2 - (address & align)) : (address & align); 

#ifdef AC_STATS
  ac_resources::ac_sim_stats.add_access(name);
#endif

  return  (ac_Hword) ((word & (0xFFFF << (byte_idx *8))) >> (byte_idx*8));
}

//!Writing a word.
void ac_storage::write( unsigned address, ac_word datum ) { 

#ifdef AC_UPDATE_LOG
  changes.push_back( change_log(address, datum , sc_simulation_time()));
#endif
  
#ifdef DEBUG_STORAGE
  cout << "Address: " << address << " PC: " << ac_resources::ac_pc << endl;
  if( !Data )
    cout << "Data is uninitialized" <<endl;

  if( address > size)
    cout << "Access out of bounds" <<endl;
#endif

  *((ac_word *)(Data+(address))) = (ac_word)datum;

#ifdef AC_STATS
  ac_resources::ac_sim_stats.add_access(name);
#endif

}

//!Writing a byte.
void ac_storage::write_byte( unsigned address, unsigned char datum ) { 

  int align = sizeof(ac_word) -1;  //Indicates the memory alignment
  int byte_idx;

  ac_word word;

  //Getting the word containing the requested byte. 
  //This is necessary because the address may not be at the beginning of a memory word.
  word = *((ac_word *)(Data+(address & ~align)));

  //Checking target endianess to discover which byte must be written
  byte_idx = (ac_resources::ac_tgt_endian) ? (align - (address & align)) : (address & align); 
  
  //Writing
  *((ac_word *)(Data+(address & ~align))) = (ac_word)((word & ~(0xFF << (byte_idx *8))) | (datum << (byte_idx*8)));

#ifdef AC_UPDATE_LOG
  changes.push_back( change_log(address, (ac_word)datum , sc_simulation_time()));
#endif

#ifdef AC_STATS
  ac_resources::ac_sim_stats.add_access(name);
#endif

}

//!Writing a short int.
void ac_storage::write_half( unsigned address, ac_Hword datum ) { 

  int align = sizeof(ac_word) -1;  //Indicates the memory alignment
  int byte_idx;

  ac_word word;

  //Getting the word containing the requested byte. 
  //This is necessary because the address may not be at the beginning of a memory word.
  word = *((ac_word *)(Data+(address & ~align)));

  //Checking target endianess to discover which byte must be written
  byte_idx = (ac_resources::ac_tgt_endian) ? (2 - (address & align)) : (address & align); 
  
  //Writing
  *((ac_word *)(Data+(address & ~align))) = (ac_word)((word & ~(0xFFFF << (byte_idx *8))) | (datum << (byte_idx*8)));

#ifdef AC_UPDATE_LOG
  changes.push_back( change_log(address, (ac_word)datum , sc_simulation_time()));
#endif

#ifdef AC_STATS
  ac_resources::ac_sim_stats.add_access(name);
#endif

}

#ifdef AC_DELAY
/*******************************
 * This methods are used for delayed assignment through write methods.
 * Normally used for memories and caches.
 */
//!Writing a word.
void ac_storage::write( unsigned address, ac_word datum, unsigned time ) { 

  delays.push_back( change_log( address, datum , time+ac_resources::ac_cycle_counter ));

}

//!Writing a byte.
void ac_storage::write_byte( unsigned address, unsigned char datum, unsigned time ) { 

  int align = sizeof(ac_word) -1;  //Indicates the memory alignment
  int byte_idx;

  ac_word word;

  //Getting the word containing the requested byte. 
  //This is necessary because the address may not be at the beginning of a memory word.
  word = *((ac_word *)(Data+(address & ~align)));

  //Checking target endianess to discover which byte must be written
  byte_idx = (ac_resources::ac_tgt_endian) ? (align - (address & align)) : (address & align); 
  
  //Delaying writing
  *((ac_word *)(Data+(address & ~align))) = (ac_word)((word & ~(0xFF << (byte_idx *8))) | (datum << (byte_idx*8)));

  delays.push_back( change_log( (address & ~align),
                                (ac_word)((word & ~(0xFF << (byte_idx *8))) | (datum << (byte_idx*8))),
                                time+ac_resources::ac_cycle_counter));
}

//!Writing a short int.
void ac_storage::write_half( unsigned address, ac_Hword datum, unsigned time ) { 

  int align = sizeof(ac_word) -1;  //Indicates the memory alignment
  int byte_idx;

  ac_word word;

  //Getting the word containing the requested byte. 
  //This is necessary because the address may not be at the beginning of a memory word.
  word = *((ac_word *)(Data+(address & ~align)));

  //Checking target endianess to discover which byte must be written
  byte_idx = (ac_resources::ac_tgt_endian) ? (2 - (address & align)) : (address & align); 
  
  //Delaying writing
  delays.push_back( change_log( (address & ~align),
                                (ac_word)((word & ~(0xFFFF << (byte_idx *8))) | (datum << (byte_idx*8))),
                                time+ac_resources::ac_cycle_counter));
}

#endif   // AC_DELAY

#ifdef AC_UPDATE_LOG

//! Reset log lists.
void ac_storage::reset_log(){ changes.clear();}
  
//!Dump storage device log.
int ac_storage::change_dump( ostream& output) {

  log_list::iterator itor;
    
  if( changes.size() ){
    output <<endl << endl;
    output << "**************** ArchC Change log *****************\n";
    output << "* Device: "<< name << "\t\t" << "PC: " << hex << ac_resources::ac_pc << dec << endl;
    output << "***************************************************\n";
    output << "*        Address         Value          Time      *\n";
    output << "***************************************************\n";
      
    for( itor = changes.begin(); itor != changes.end(); itor++)  
      output << "*  " << *itor << "     *" << endl;
      
    output << "***************************************************\n";
  }
  return 0;
}
    
//!Save storage device log.
void ac_storage::change_save( ) {

  log_list::iterator itor;
    
  if( changes.size() ){

    for( itor = changes.begin(); itor != changes.end(); itor++)  
      itor->save(this->update_file);
      
  }
}
    
#endif



//!Method to load  content from a file.
void ac_storage::load( char* file ){

  ifstream input;
  string read;
  string word;
  istringstream line;
  unsigned text_size=0;
  extern unsigned dec_cache_size;
  bool is_addr, is_text=0, first_addr=1;
  long long data;
  unsigned int  addr=0;

  //Try to read as ELF first
  extern int  ac_load_elf(char*, char*, unsigned int);
  if (ac_load_elf(file, Data, size) == EXIT_SUCCESS) {
    //init decode cache and return
    if(!dec_cache_size)
      dec_cache_size = ac_heap_ptr;
    return;
  }

  // Looking for initialization file.
  input.open(file);
  if(!input){
    AC_ERROR("Could not open input file:" << file);
    AC_ERROR("Simulation aborted.");
    exit(1);
  }
  else{
      
    while( !input.eof() ){

      line.clear();
      getline(input, read);
      line.str(read);
                        
      is_addr = 1;

      //Processing line
      while(line >> word){

        if( word == ".text:" ){
          is_text = 1;
          continue;
        }
                                                
        if( word[0] == '.' ){
          is_text = 0;
          continue;
        }

        //Processing word
        if( is_addr ){
          addr = strtol(word.c_str(), NULL, 16);
          is_addr = 0;
          if( is_text && first_addr ){
            first_addr = 0;
            ac_resources::ac_start_addr = addr;
          }
        }
        else{
                                        
          if(is_text)text_size++;
          data = strtoll( word.c_str(), NULL, 16);
          write(addr,(ac_word)data);
          addr+= AC_WORDSIZE/8;

          //keep the maximum address in dec_cache_size
          if (dec_cache_size < addr) dec_cache_size = addr;

        }
      }
    }
  }

  input.close();

}


//!Method to load content from a array.
void ac_storage::load_array( const unsigned char* d, const unsigned s )
{
  if (size < s) {
    fprintf(stderr, "Storage %s: trying to load an array bigger then storage size.\n", name);
    exit(EXIT_FAILURE);
  }
  memcpy(Data, d, s);
}



#ifdef AC_DELAY

//!Method to commit delayed updates.
void ac_storage::commit_delays( double time ){
  log_list::iterator itor;
    
  itor = delays.begin();

  //Sometimes, when a memory hierarchy is present and the processor spends some
  //cycles in a wait status, we may have to commit changes for every cycle <= 
  //current time.
  while( delays.size() && (itor->time <= time) ){
    write( itor->addr, itor->value );
    itor = delays.erase( itor);
  }
}
#endif

                
