/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */

/*  ArchC Storage Library for the ArchC architecture simulators
    Copyright (C) 2002-2004  The ArchC Team

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.
*/

/********************************************************/
/* The ArchC storage device base class.                 */
/* Author:  Sandro Rigo                                 */
/*                                                      */
/*                                                      */
/* The ArchC Team                                       */
/* Computer Systems Laboratory (LSC)                    */
/* IC-UNICAMP                                           */
/* http://www.lsc.ic.unicamp.br                         */
/********************************************************/

//////////////////////////////////////////////////////////
/*!\file ac_storage.H
  \brief The ArchC storage device base class.

  This class contains  basic members and methods needed by
  all storage devices for behavioral simulation. 
*/
//////////////////////////////////////////////////////////
#ifndef _AC_STORAGE_H
#define _AC_STORAGE_H

#include <stdio.h>
#include <stdlib.h>
#include <systemc.h>
#include <list>
#include <fstream>
#include "archc.H"
#include "ac_parms.H"

using std::string;
using std::istringstream;

/////////////////////////////////////////////////////
/*!ArchC class for storage devices                 */
/////////////////////////////////////////////////////
class ac_storage { 

protected:
  char *Data;                       //!< The data stored into the device
  char *name;                       //!< Device name
  unsigned size;                    //!< Device size

#ifdef AC_UPDATE_LOG
  log_list changes;                 //!< Update log.
  fstream update_file;              //!< Update log binary file.
#endif


#ifdef AC_DELAY
  log_list delays;                  //!< Delayed update queue.
#endif

public:
  
  //!Reading a word
  virtual ac_word read( unsigned address );

  //!Reading a byte
  virtual unsigned char read_byte( unsigned address );

  //!Reading half word
  virtual ac_Hword read_half( unsigned address );


  //!Writing a word
  virtual void write( unsigned address, ac_word datum );

  //!Writing a byte 
  virtual void write_byte( unsigned address, unsigned char datum );

  //!Writing a short int 
  virtual void write_half( unsigned address, ac_Hword datum );

#ifdef AC_DELAY
  //!Writing a word
  virtual void write( unsigned address, ac_word datum, unsigned time );

  //!Writing a byte 
  virtual void write_byte( unsigned address, unsigned char datum, unsigned time );

  //!Writing a short int 
  virtual void write_half( unsigned address, ac_Hword  datum, unsigned time  );
#endif

#ifdef AC_UPDATE_LOG

  //! Reset log lists.
  void reset_log();
  
  //!Dump storage device log.
  int change_dump( ostream& output);


  //!Save storage device log.
  void change_save( );
#endif
      
  //!Load a binary image file
  void load_binary( const char* progmem )
  {
    FILE *file;
    unsigned long fileLen;

    //Open file
    file = fopen(progmem, "rb");
    if (!file) {
      AC_ERROR("Storage " << name << ": Unable to open binary memory copy from file " << progmem);
      return;
    }

    //Get file length
    fseek(file, 0, SEEK_END);
    fileLen=ftell(file);
    fseek(file, 0, SEEK_SET);

    //Test size
    if (size < fileLen) {
      AC_ERROR("Storage " << name << ": trying to load a file (" << fileLen << " bytes) bigger then storage size (" << size << " bytes)");
      fclose(file);
      exit(EXIT_FAILURE);
    }

    //Read file contents into buffer
    fread(Data, fileLen, 1, file);
    fclose(file);
  }

  //!Constructor.
  ac_storage( char *n, unsigned s ){
      
    //Initializing members
    Data = (char*) new char[ s ];
    name = n;
    size = s;

#ifdef AC_UPDATE_LOG

    //char filename[strlen(n)+ 16];

    //sprintf(filename, "ac_storage.%s.log",n);
    //update_file.open(filename, ios::out|ios::binary);
#endif

  }    
  
  //!Constructor. Without Data Initialization. 
  /*!Used to declare devices where sizes will be computed dynamically*/
  ac_storage( char *n ):name(n){};

  //!Destructor.
  virtual ~ac_storage(){

#ifdef AC_UPDATE_LOG
    update_file.close();
#endif

    //Freeing Memory.
    delete[] Data;
  }    

  //!Copy operator
  ac_storage& operator =(ac_storage& src ){
    
    Data = src.Data;
    name = src.name;
    size = src.size;
    
#ifdef AC_UPDATE_LOG
    changes = src.changes;
#endif
    return *this;
  }

  //!Method to initialize Data array dynamically
  void initialize( char* n, unsigned s ){
    name = n;
    Data = (char*) new char[ s ];
    size = s;
  }

  //!Method to provide the name of the device.
  char* get_name( ){
    return name;
  }

  //!Method to provide the size of the device.
  unsigned get_size( ){
    return size;
  }

#ifdef AC_UPDATE_LOG
  //!Method to provide the change list.
  log_list* get_changes( ){
    return &changes;
  }
#endif

  //!Method to load device content from a file.
  void load( char* file );

  //!Method to load device content from an array.
  void load_array( const unsigned char* d, const unsigned s );


#ifdef AC_DELAY

  //!Commiting delayed updates
  virtual void commit_delays( double time );

#endif
    
  //!Dump the entire contents of a storage device
  virtual void dump(){
    fstream dfile;              //!< Dump file.
    unsigned i;

    char filename[strlen(name)+ 6];

    sprintf(filename, "%s.dump",name);
    dfile.open(filename, ios::out);

    for( i=0; i<size;i+=(sizeof(ac_word))){
      dfile << hex << i << "  " << read(i) <<endl;
    }
    dfile.close();
  }
};


///////////////////////////////////
//!Overloaded operators          //
///////////////////////////////////
// unsigned operator = ( unsigned d, ac_reg r){ 

//      unsigned u = d;
    
//      u = r.read();

//      return u;
// }


#endif /*_AC_STORAGE_H*/
