/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */

/*  ArchC Storage Library for the ArchC architecture simulators
    Copyright (C) 2002-2004  The ArchC Team

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.
*/

/********************************************************/
/* The ArchC storage device base class.                 */
/* Author:  Sandro Rigo                                 */
/*                                                      */
/*                                                      */
/* The ArchC Team                                       */
/* Computer Systems Laboratory (LSC)                    */
/* IC-UNICAMP                                           */
/* http://www.lsc.ic.unicamp.br                         */
/********************************************************/

//////////////////////////////////////////////////////////
/*!\file ac_storage.H
  \brief The ArchC storage device base class.

  This class contains  basic members and methods needed by
  all storage devices for behavioral simulation. 
*/
//////////////////////////////////////////////////////////
#ifndef _AC_REGBANK_H
#define _AC_REGBANK_H

#include <sstream>
#include <string>
#include <fstream>
#include <iterator>
#include "ac_storage.H"

extern bool& ac_wait_sig;
extern double& time_step;
  
//////////////////////////////////////
//! ArchC class specialized in modeling register bank.
template<class T>class ac_regbank: public ac_storage {
public:
  
  //!Return the reference of an address.
  ac_word& operator[] ( unsigned address ){
#ifdef AC_STATS  
    ac_sim_stats.add_access(name);
#endif

    return ((ac_word*) Data)[address]; 
  }

  //!Reading the content of an address. Overloaded Method.
  ac_word read( unsigned address ){

#ifdef AC_STATS  
    ac_sim_stats.add_access(name);
#endif

    return ((ac_word*) Data)[address]; 
  }

  //!Reading the content of an address. Overloaded Method.
  ac_Dword read_double( unsigned address ){

#ifdef AC_STATS  
    ac_sim_stats.add_access(name);
#endif

    return ((ac_Dword*) Data)[address]; 
  }

  void write_double( unsigned address , ac_Dword datum  ){
#ifdef AC_MEM_HIERARCHY
    if(ac_wait_sig)
      return;
#endif
  
#ifdef AC_UPDATE_LOG
    changes.push_back( change_log(address, datum , sc_simulation_time()));
#endif
  
    ((ac_Dword*) Data)[address] = datum;
  
#ifdef AC_STATS
    ac_sim_stats.add_access(name);
#endif
  }

  //!Writing to a register. Overloaded Method.
  void write( unsigned address , T datum  ){
#ifdef AC_MEM_HIERARCHY
    if(ac_wait_sig)
      return;
#endif
  
#ifdef AC_UPDATE_LOG
    changes.push_back( change_log(address, datum , sc_simulation_time()));
#endif
  
    ((T *) Data)[address] = datum;
  
#ifdef AC_STATS
    ac_sim_stats.add_access(name);
#endif
  }


#ifdef AC_DELAY
  //!Writing to a register. Overloaded Method.
  void write( unsigned address , T datum, unsigned time ){
    delays.push_back( change_log( address, datum, (time * time_step) + time_step + sc_simulation_time()));
  }
#endif

  //!Constructor
  ac_regbank( char* name, unsigned size ):ac_storage(name,size*sizeof(T)){}

  //!Dump the entire contents of a regbank device
  void dump(){
    fstream dfile;              //!< Dump file.
    unsigned i;

    char filename[strlen(name)+ 6];

    sprintf(filename, "%s.dump",name);
    dfile.open(filename, ios::out);

    for( i=0; i<(size/sizeof(T));i++){
      dfile << dec << i << hex << "  " << read(i) <<endl;
    }
    dfile.close();
  }

  //!Method to load device content from a file.
  void load( char* file ){

    ifstream input;
    string read;
    string word;
    T data;
    istringstream line;

    bool is_addr;

    int  addr=0;

    // Looking for initialization file.
    input.open(file);
    if(!input){
      AC_ERROR("Could not open input file:" << file);
    }
    else{
      
      while( !input.eof() ){

        line.clear();
        getline(input, read);
        line.str(read);
                        
        is_addr = 1;

        //Processing line
        while(line >> word){

          if( word == ".text" ){
            AC_ERROR("Should not load code into the register bank: " << name);
            return;
          }
                                                
          if( word == ".data" ){
            continue;
          }

          //Processing word
          if( is_addr ){
            addr = strtol(word.c_str(), NULL, 16);
            is_addr = 0;
          }
          else{
            data = (T)strtol( word.c_str(), NULL, 16);
            write(addr,data);
            addr+= sizeof(T);
          }
        }
      }
    }
  }
};

#endif //_AC_REGBANK_H




