/**
 * @file      arm7-isa.cpp
 * @author    André Luiz Pasquali
 *            
 *            The ArchC Team
 *            http://www.archc.org/
 *
 *            Computer Systems Laboratory (LSC)
 *            IC-UNICAMP
 *            http://www.lsc.ic.unicamp.br
 *
 * @version   1.0
 * @date      Mon, 19 Jun 2006 15:33:28 -0300
 * 
 * @brief     The ArchC R3000 cycle-accurate model.
 * 
 * @attention Copyright (C) 2002-2006 --- The ArchC Team
 * 
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version. 
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details. 
 * 
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include "archc.H"
#include "ac_types.H"
#include "ac_parms.H"
#include  "arm7-isa.H"
#include  "ac_isa_init.cpp"


//!User defined macros to reference registers.
#define LR 14  //linker register
#define PC 15  //program counter

// Forwarded values
ac_Uword ex_value1, ex_value2, shifter_carry_out;
ac_Uword id_value1, id_value2;
bool execute= true;

ac_Uword RotateRight(int rotateamount, ac_Uword reg);
ac_Uword SignExtend(ac_Uword bitSeq, int bitSeq_length);
void ceckCond(ac_Uword n, ac_Uword z, ac_Uword c, ac_Uword v );


/************************************************************************/
//!Generic instruction behavior method.

void ac_behavior( instruction ){

  switch( stage ) {
  case IF:
    ac_pc += 4;
    RB.write(PC, ac_pc);
    IF_ID.npc = ac_pc;
    break;
    
  case ID:
    break;

  case EX:
    ceckCond(n, z, c, v);
    if(!execute)
	return;

    break;

  default:
    break;
  }
};
/************************************************************************/

/************************************************************************/
//! Common behaviour for all Type_DataProcessing instructions

void ac_behavior( Type_DataProcessing ){
 
  switch( stage ) {
  case IF:
    break;

  case ID:
    ID_EX.npc = IF_ID.npc;
    ID_EX.rn = rn;
    ID_EX.rd = rd;
    ID_EX.data1 = RB.read(rn);
    
    if(i == 1)
      ID_EX.data2 = (RotateRight(rat*2, imm));
    else 		 	    
      ID_EX.data2 = RB.read(rm);
      shifter_carry_out = CPSR.c;

    break;

  case EX:
    break;

  default:
    break;
  }
}
/************************************************************************/

/************************************************************************/
//! Common behaviour for all Type_Multiply instructions
void ac_behavior( Type_Multiply ){

  switch( stage ) {
  case IF:
    break;

  case ID:
    ID_EX.npc = IF_ID.npc;
    ID_EX.rd = rdM;
    if(a==0){
    	ID_EX.data1 = RB.read(rsM);
    	ID_EX.data2 = RB.read(rm);
    }
    else{
   	ID_EX.data1 = RB.read(rsM);
    	ID_EX.data2 = RB.read(rm);
    	ID_EX.data3 = RB.read(rnM);
    }
    break;

  case EX:
    break;

  default:
    break;
  }
}
/************************************************************************/


/************************************************************************/
//! Common behaviour for all Type_DataTransfer instructions

void ac_behavior( Type_DataTransfer ){

  switch( stage ) {
  case IF:
    break;

  case ID:
    ID_EX.npc = IF_ID.npc;
    ID_EX.rd = rd;
    ID_EX.address = address;
    
    break;

  case EX:
    break;

  default:
    break;
  }
}
/************************************************************************/

/************************************************************************/
//! Common behaviour for all Type_BranchExchange instructions

void ac_behavior( Type_BranchExchange ){

  switch( stage ) {
  case IF:
    break;

  case ID:
	ID_EX.npc = IF_ID.npc;
	if(code == 0x05){
	  ID_EX.signed_immed = signed_immed;
          ID_EX.lb = lb;  
        }
        else if(code2 == 0x12)
          ID_EX.data1 = RB.read(rm);
		 	   
    break;

  case EX:
    break;

  default:
    break;
  }
}
/************************************************************************/

///Performs a right bit rotation
///@param rotateamount the number of bit to shift
///@param reg the quantity which must be shifted
///@return the rotated quantity
ac_Uword RotateRight(int rotateamount, ac_Uword reg) {
	ac_Uword temp = reg;
	
	if(rotateamount <= 0){
		shifter_carry_out = CPSR.c;
		return temp;
	}	
	//Masks
	unsigned long mask1 = ((unsigned long)0xFFFFFFFF) >> (32 - rotateamount);
	unsigned long mask2 = ((unsigned long)0xFFFFFFFF) >> rotateamount;
	//value which must be glued to the left of the shifted quantity
	unsigned long buff = temp & mask1; 
	
	temp = (temp >> rotateamount) & mask2;
	buff <<= (32 - rotateamount);
	temp |= buff;
	shifter_carry_out = temp << 31;
	return temp;
}
// estender o bitseq para 32 bits
ac_Uword SignExtend(ac_Uword bitSeq, int bitSeq_length){
	if(bitSeq_length <= 0)
		return bitSeq;
	
	if((bitSeq & (1 << bitSeq_length - 1)) != 0)
		bitSeq |= (((unsigned long)0xFFFFFFFF) << bitSeq_length);
	else 
		bitSeq &= (((unsigned long)0xFFFFFFFF) >> (32 - bitSeq_length));
	
	return bitSeq;
}

void copySPSR(){
   CPSR.n = SPSR.n;
   CPSR.z = SPSR.z;
   CPSR.c = SPSR.c;
   CPSR.v = SPSR.v;
   CPSR.Irq = SPSR.Irq;
   CPSR.Fiq = SPSR.Fiq;
   CPSR.t = SPSR.t;
   CPSR.M4 = SPSR.M4;
   CPSR.M3 = SPSR.M3;
   CPSR.M2 = SPSR.M2;
   CPSR.M1 = SPSR.M1;
}

void ceckCond(ac_Uword n, ac_Uword z, ac_Uword c, ac_Uword v ){
        execute= false;
        //EQ
	if(n==0 && z == 0 && c == 0 && v == 0 ){
		if(CPSR.z == 0x1)		
		   execute = true;
		return;	
	}
        //NE 
	else if(n==0 && z == 0 && c == 0 && v == 1 ){
		if(CPSR.z == 0x0)		
		   execute = true;
		return;	
	}

	//CS/HS
	else if(n==0 && z == 0 && c == 1 && v == 0 ){
		if(CPSR.c == 0x1)		
		execute = true;
		return;	
	}

	//CC/LO
	else if(n==0 && z == 0 && c == 1 && v == 1 ){
		if(CPSR.c == 0x0)		
		execute = true;
		return;	
	}

	//MI
	else if(n==0 && z == 1 && c == 0 && v == 0 ){
		if(CPSR.n == 0x1)		
		execute = true;
		return;	
	}

	//PL
	else if(n==0 && z == 1 && c == 0 && v == 1 ){
		if(CPSR.n == 0x0)		
		execute = true;
		return;	
	}

	//VS
	else if(n==0 && z == 1 && c == 1 && v == 0 ){
		if(CPSR.v == 0x1)		
		execute = true;
		return;	
	}

	//VC
	else if(n==0 && z == 1 && c == 1 && v == 1 ){
		if(CPSR.v == 0x0)		
		execute = true;
		return;	
	}

	//HI
	else if(n==1 && z == 0 && c == 0 && v == 0 ){
		if(CPSR.c == 0x1 && CPSR.z == 0x0)		
		execute = true;
		return;	
	}

	//LS
	else if(n==1 && z == 0 && c == 0 && v == 1 ){
		if(CPSR.c == 0x0 || CPSR.z == 0x1)		
		execute = true;
		return;	
	}

	//GE
	else if(n==1 && z == 0 && c == 1 && v == 0 ){
		if(CPSR.n == CPSR.v)		
		execute = true;
		return;	
	}

	//LT
	else if(n==1 && z == 0 && c == 1 && v == 1 ){
		if(CPSR.n != CPSR.v)		
		execute = true;
		return;	
	}

	//GT
	else if(n==1 && z == 1 && c == 0 && v == 0 ){
		if(CPSR.n == CPSR.v && CPSR.z == 0)		
		execute = true;
		return;	
	}

	//LE
	else if(n==1 && z == 1 && c == 0 && v == 1 ){
		if(CPSR.n != CPSR.v && CPSR.z == 1)		
		execute = true;
		return;	
	}

	//AL
	else if(n==1 && z == 1 && c == 1 && v == 0 ){
		execute = true;
		return;	
	}

	//NV
	else if(n==1 && z == 1 && c == 1 && v == 1 ){
		execute = false;
		return;	
	}

}

//!Instruction add behavior method.
void ac_behavior( add ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
    
   ex_value1 =	ID_EX.data1 + ID_EX.data2;

  if(s==1 && ID_EX.rd == 15)
       copySPSR();
  else if(s==1){
	CPSR.n = ID_EX.rd >> 31;
        if(ID_EX.rd==0)
           CPSR.z=0x1;
        else 
	   CPSR.z=0x0;
	
	CPSR.c = ((ID_EX.data1 & ID_EX.data2 & 0x80000000) | (((~ID_EX.data1 | ~ID_EX.data2) & ex_value1) & 0x80000000) );	
       	CPSR.v = ((ID_EX.data1 & ID_EX.data2 & ~ex_value1 & 0x80000000) | (~ID_EX.data1 & ~ID_EX.data2 & ex_value1 & 0x80000000));	

   }
  
  
   RB.write(ID_EX.rd, ex_value1);	
  default:
    break;
  }
};

//!Instruction and1 behavior method.
void ac_behavior( and1 ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
   ex_value1 =	ID_EX.data1 & ID_EX.data2;
  
 
   if(s==1 && ID_EX.rd == 15)
       copySPSR();
   else if(s==1){
	CPSR.n = ID_EX.rd >> 31;
        if(ID_EX.rd==0)
           CPSR.z=0x1;
        else 
	   CPSR.z=0x0;
	
	CPSR.c = ((ID_EX.data1 & ID_EX.data2 & 0x80000000) | (((~ID_EX.data1 | ~ID_EX.data2) & ex_value1) & 0x80000000) );	
       
   }

   RB.write(ID_EX.rd, ex_value1);
   
    break;

  default:
    break;
  }
};

//!Instruction eor behavior method.
void ac_behavior( eor ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
   ex_value1 =	ID_EX.data1 ^ ID_EX.data2;
  
    if(s==1 && ID_EX.rd == 15)
       copySPSR();
   else if(s==1){
	CPSR.n = ID_EX.rd >> 31;
        if(ID_EX.rd==0)
           CPSR.z=0x1;
        else 
	   CPSR.z=0x0;
	
	CPSR.c = ((ID_EX.data1 & ID_EX.data2 & 0x80000000) | (((~ID_EX.data1 | ~ID_EX.data2) & ex_value1) & 0x80000000) );	
       
   }

   RB.write(ID_EX.rd, ex_value1);	
     
   
    break;

  default:
    break;
  }
};

//!Instruction sub behavior method.
void ac_behavior( sub ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
   ex_value1 =	ID_EX.data1 - ID_EX.data2;
 
 
   if(s==1 && ID_EX.rd == 15)
       copySPSR();
   else if(s==1){
	CPSR.n = ID_EX.rd >> 31;
        if(ID_EX.rd==0)
           CPSR.z=0x1;
        else 
	   CPSR.z=0x0;
	
	CPSR.c = ~((ID_EX.data1 & ID_EX.data2 & 0x80000000) | (((~ID_EX.data1 | ~ID_EX.data2) & ex_value1) & 0x80000000) );	
       	CPSR.v = ((ID_EX.data1 & ID_EX.data2 & ~ex_value1 & 0x80000000) | (~ID_EX.data1 & ~ID_EX.data2 & ex_value1 & 0x80000000));	

   }	 
  

   RB.write(ID_EX.rd, ex_value1);	
   
    break;

  default:
    break;
  }
};

//!Instruction rsb behavior method.
void ac_behavior( rsb ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
   ex_value1 =	ID_EX.data2 - ID_EX.data1 ;
  
   
   if(s==1 && ID_EX.rd == 15)
       copySPSR();
   else if(s==1){
	CPSR.n = ID_EX.rd >> 31;
        if(ID_EX.rd==0)
           CPSR.z=0x1;
        else 
	   CPSR.z=0x0;
	
	CPSR.c = ~((ID_EX.data1 & ID_EX.data2 & 0x80000000) | (((~ID_EX.data1 | ~ID_EX.data2) & ex_value1) & 0x80000000) );	
       	CPSR.v = ((ID_EX.data1 & ID_EX.data2 & ~ex_value1 & 0x80000000) | (~ID_EX.data1 & ~ID_EX.data2 & ex_value1 & 0x80000000));	

   }	 
  
   RB.write(ID_EX.rd, ex_value1);	
   
    break;

  default:
    break;
  }
};

//!Instruction adc behavior method.
void ac_behavior( adc ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
   ac_Uword carry;
   carry = (0x00000000 | CPSR.c) << 31;
   ex_value1 =	ID_EX.data1 + ID_EX.data2 + carry;

  if(s==1 && ID_EX.rd == 15)
       copySPSR();
  else if(s==1){
	CPSR.n = ID_EX.rd >> 31;
        if(ID_EX.rd==0)
           CPSR.z=0x1;
        else 
	   CPSR.z=0x0;
	
	CPSR.c = ((ID_EX.data1 & ID_EX.data2 & 0x80000000) | (((~ID_EX.data1 | ~ID_EX.data2) & ex_value1) & 0x80000000) );	
       	CPSR.v = ((ID_EX.data1 & ID_EX.data2 & ~ex_value1 & 0x80000000) | (~ID_EX.data1 & ~ID_EX.data2 & ex_value1 & 0x80000000));	

  }
  
  
   RB.write(ID_EX.rd, ex_value1);
   
    break;

  default:
    break;
  }
};

//!Instruction sbc behavior method.
void ac_behavior( sbc ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
   ac_Uword carry;
   carry = (0x00000000 | CPSR.c) << 31;
   ex_value1 = ID_EX.data1 - ID_EX.data2 - ~carry;
 

   if(s==1 && ID_EX.rd == 15)
       copySPSR();
   else if(s==1){
	CPSR.n = ID_EX.rd >> 31;
        if(ID_EX.rd==0)
           CPSR.z=0x1;
        else 
	   CPSR.z=0x0;
	
	CPSR.c = ~((ID_EX.data1 & ID_EX.data2 & 0x80000000) | (((~ID_EX.data1 | ~ID_EX.data2) & ex_value1) & 0x80000000) );	
       	CPSR.v = ((ID_EX.data1 & ID_EX.data2 & ~ex_value1 & 0x80000000) | (~ID_EX.data1 & ~ID_EX.data2 & ex_value1 & 0x80000000));	

   }	 

   RB.write(ID_EX.rd, ex_value1);
   
    break;

  default:
    break;
  }
};

//!Instruction rsc behavior method.
void ac_behavior( rsc ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
   ac_Uword carry;
   carry = (0x00000000 | CPSR.c) << 31;
   ex_value1 =	ID_EX.data2 - ID_EX.data1 -~carry;
 
   
  if(s==1 && ID_EX.rd == 15)
       copySPSR();
   else if(s==1){
	CPSR.n = ID_EX.rd >> 31;
        if(ID_EX.rd==0)
           CPSR.z=0x1;
        else 
	   CPSR.z=0x0;
	
	CPSR.c = ~((ID_EX.data1 & ID_EX.data2 & 0x80000000) | (((~ID_EX.data1 | ~ID_EX.data2) & ex_value1) & 0x80000000) );	
       	CPSR.v = ((ID_EX.data1 & ID_EX.data2 & ~ex_value1 & 0x80000000) | (~ID_EX.data1 & ~ID_EX.data2 & ex_value1 & 0x80000000));	

   }	 
	 
    
   RB.write(ID_EX.rd, ex_value1);
   
    break;

  default:
    break;
  }
};

//!Instruction tst behavior method.
void ac_behavior( tst ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
   
   ex_value1 =	ID_EX.data2 & ID_EX.data1;
  
   
  if(s==1 && ID_EX.rd == 15)
       copySPSR();
   else if(s==1){
	CPSR.n = ex_value1 >> 31;
        if(ex_value1==0)
           CPSR.z=0x1;
        else 
	   CPSR.z=0x0;
	
	CPSR.c = shifter_carry_out;
      
   }	 
   
    break;

  default:
    break;
  }
};

//!Instruction teq behavior method.
void ac_behavior( teq ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
   ex_value1 =	ID_EX.data2 ^ ID_EX.data1;
  
   
  if(s==1 && ID_EX.rd == 15)
       copySPSR();
   else if(s==1){
	CPSR.n = ex_value1 >> 31;
        if(ex_value1==0)
           CPSR.z=0x1;
        else 
	   CPSR.z=0x0;
	
	CPSR.c = shifter_carry_out;
      
   }	 
   break;

  default:
    break;
  }
};

//!Instruction cmp behavior method.
void ac_behavior( cmp ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
  
   ex_value1 =	ID_EX.data1 - ID_EX.data2;
  
   
  CPSR.n = ex_value1 >> 31;
  if(ex_value1==0)
  	CPSR.z=0x1;
  else 
	CPSR.z=0x0;
	
   CPSR.c = ~((ID_EX.data1 & ID_EX.data2 & 0x80000000) | (((~ID_EX.data1 | ~ID_EX.data2) & ex_value1) & 0x80000000) );	
   CPSR.v = ((ID_EX.data1 & ID_EX.data2 & ~ex_value1 & 0x80000000) | (~ID_EX.data1 & ~ID_EX.data2 & ex_value1 & 0x80000000));	

   	 
    break;

  default:
    break;
  }
};

//!Instruction cmn behavior method.
void ac_behavior( cmn ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:

   ex_value1 =	ID_EX.data1 + ID_EX.data2;
 
   
  CPSR.n = ex_value1 >> 31;
  if(ex_value1==0)
  	CPSR.z=0x1;
  else 
	CPSR.z=0x0;
	
  CPSR.c = ((ID_EX.data1 & ID_EX.data2 & 0x80000000) | (((~ID_EX.data1 | ~ID_EX.data2) & ex_value1) & 0x80000000) );	
  CPSR.v = ((ID_EX.data1 & ID_EX.data2 & ~ex_value1 & 0x80000000) | (~ID_EX.data1 & ~ID_EX.data2 & ex_value1 & 0x80000000));	

  	 
    break;

  default:
    break;
  }
};

//!Instruction orr behavior method.
void ac_behavior( orr ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
   ex_value1 =	ID_EX.data1 | ID_EX.data2 ;
 
	 
   if(s==1 && ID_EX.rd == 15)
       copySPSR();
   else if(s==1){
	CPSR.n = ID_EX.rd >> 31;
        if(ID_EX.rd==0)
           CPSR.z=0x1;
        else 
	   CPSR.z=0x0;
	
	CPSR.c = shifter_carry_out;	
  } 
   RB.write(ID_EX.rd, ex_value1);
   
    break;

  default:
    break;
  }
};

//!Instruction mov behavior method.
void ac_behavior( mov ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
 
   RB.write(ID_EX.rd, ID_EX.data1);
   
   if(s==1 && ID_EX.rd == 15)
       copySPSR();
   else if(s==1){
	CPSR.n = ID_EX.rd >> 31;
        if(ID_EX.rd==0)
           CPSR.z=0x1;
        else 
	   CPSR.z=0x0;
	
	CPSR.c = shifter_carry_out;	
  }	 
 
  break;

  default:
    break;
  }
};

//!Instruction bic behavior method.
void ac_behavior( bic ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
   ex_value1 =	ID_EX.data1 & ~ID_EX.data2 ;
  
   if(s==1 && ID_EX.rd == 15)
       copySPSR();
   else if(s==1){
	CPSR.n = ID_EX.rd >> 31;
        if(ID_EX.rd==0)
           CPSR.z=0x1;
        else 
	   CPSR.z=0x0;
	
	CPSR.c = shifter_carry_out;	
  }	  
  
   RB.write(ID_EX.rd, ex_value1);
   
    break;

  default:
    break;
  }
};

//!Instruction mvn behavior method.
void ac_behavior( mvn ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
   ex_value1 = ~ID_EX.data2 ;
  

   if(s==1 && ID_EX.rd == 15)
       copySPSR();
   else if(s==1){
	CPSR.n = ID_EX.rd >> 31;
        if(ID_EX.rd==0)
           CPSR.z=0x1;
        else 
	   CPSR.z=0x0;
	
	CPSR.c = shifter_carry_out;	
  }	 
    
   RB.write(ID_EX.rd, ex_value1);
   
    break;

  default:
    break;
  }
};

//!Instruction mla behavior method.
void ac_behavior( mla ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
   ex_value1 = ID_EX.data3 + (ID_EX.data1*ID_EX.data2) ;
  
   
   if(s==1 && ID_EX.rd == 15)
       copySPSR();
   else if(s==1){
	CPSR.n = ID_EX.rd >> 31;
        if(ID_EX.rd==0)
           CPSR.z=0x1;
        else 
	   CPSR.z=0x0;
   }	 
 
   RB.write(ID_EX.rd, ex_value1);
   
    break;

  default:
    break;
  }
};

//!Instruction mul behavior method.
void ac_behavior( mul ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
   ex_value1 = ID_EX.data1 * ID_EX.data2;
  
   
   if(s==1 && ID_EX.rd == 15)
       copySPSR();
   else if(s==1){
	CPSR.n = ID_EX.rd >> 31;
        if(ID_EX.rd==0)
           CPSR.z=0x1;
        else 
	   CPSR.z=0x0;
	
  }	 
   
   RB.write(ID_EX.rd, ex_value1);
   
   break;

  default:
    break;
  }
};


//!Instruction ldr behavior method.
void ac_behavior( ldr ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
   
   ex_value1 = DM.read(ID_EX.address & 0xFFFFFFFC);
   if(ID_EX.rd == PC){
       RB.write(PC, ex_value1 & 0xFFFFFFFE);
   }
   else 
       RB.write(ID_EX.rd, ex_value1);	
    
    break;

  default:
    break;
  }
};

//!Instruction str behavior method.
void ac_behavior( str ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
    
    DM.write(ID_EX.address & 0xFFFFFFFC, RB.read(ID_EX.rd));
    break;

  default:
    break;
  }
};

//!Instruction bl behavior method.
void ac_behavior( bl ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:
    break;

  case EX:
    if(ID_EX.lb == 0x1)
	RB.write(LR, RB.read(PC));
    ex_value1 = RB.read(PC) + (SignExtend( ID_EX.signed_immed, 24) << 2);
    break;

  default:
    break;
  }
};

//!Instruction bx behavior method.
void ac_behavior( bx ){
  
  switch( stage ) {
  case IF:
    /* It was implemented for all instructions on ac_behaviour( instruction ) */

    break;

  case ID:

    break;

  case EX:
    ex_value1 = ID_EX.data1 & 0XFFFFFFFE;
    RB.write(PC, ex_value1);
    break;

  default:
    break;
  }
};

//!Instruction sys_call behavior method.
void ac_behavior( sys_call ) {

  switch( stage ) {
  case IF:
    break;
    
  case ID:
    break;
    
  case EX:
    ac_stop();
    break;
    
  default:
    break;
  }
};



