<?php
class CommonHelper extends AppHelper {

    public $helpers = ['Locale.Locale'];

/**
 * Cache dos modelos já carregados no helper
 * @var array
 */
    protected $_models = [];

/**
 * Cache das introspecções nos campos de cada modelo
 *
 * @var array
 */
    public $fieldset = [];

/**
 * Apartir dos dados de uma busca ($data), monta uma linha de uma tabela
 * para as ações 'Index' que utilizam o widget 'table'.
 *
 * @param  array  $data          Dados de uma entrada dentre as retornadas por um find All
 * @param  string $defaultModel  Nome do modelo padrão
 * @param  array  $includeFields Lista com o nome das colunas que serão incluídas na linha
 * @return array
 */
    public function buildBodyIndexRow($data, $defaultModel, $includeFields)
    {
        $row = [];
        $meta = $this->_introspectModel($defaultModel);

        foreach ($includeFields as $field) {
            if (strpos($field, '.') !== false) {
                $row[$field] = Hash::get($data, $field);
                continue;
            }

            $row[$field] = $data[$defaultModel][$field];

            if (isset($meta[$field])) {
                $row[$field] = $this->_formatDefaults($row[$field], $meta[$field]['type']);
            }

            if ($field === 'cellphone') {
                $row[$field] = $this->formatPhoneNumber($row[$field]);
                continue;
            }

            if (preg_match('/_id$/', $field) === 0) {
                continue;
            }

            $modelName = Inflector::camelize(substr($field, 0, -3));
            if (!isset($data[$modelName])) {
                continue;
            }

            if (isset($data[$modelName]['alias'])) {
                $row[$field] = $data[$modelName]['alias'];
                continue;
            }

            if (isset($data[$modelName]['name'])) {
                $row[$field] = $data[$modelName]['name'];
                continue;
            }

            if (isset($data[$modelName]['title'])) {
                $row[$field] = $data[$modelName]['title'];
                continue;
            }
        }

        return $row;
    }

/**
 * Formata $value fazendo instrospecção no tipo de coluna
 * do campo $field do model $modelName
 *
 * @param  string $field
 * @param  mixed  $value
 * @param  strin  $modelName
 * @return mixed
 */
    public function magicFormat($field, $value, $modelName = null)
    {
        $meta = $this->_introspectModel($modelName);

        if (isset($meta[$field])) {
            return $this->_formatDefaults($value, $meta[$field]['type']);
        }

        return $value;
    }

/**
 * Formata a saída automaticamente de acordo com o tipo do
 * campo no banco de dados.
 *
 * @param  mixed  $value Valor vindo do BD
 * @param  string $type  Tipo do campo no BD
 * @return
 */
    protected function _formatDefaults($value, $type)
    {
        switch ($type) {
            case 'boolean':
                return $value ? __('Sim') : __('Não');

            case 'float':
            case 'decimal':
            case 'numeric':
                return $this->Locale->number($value);

            case 'date':
                return $this->Locale->date($value);

            case 'datetime':
                return $this->Locale->dateTime($value);

            case 'text':
                return '<br>' . nl2br($value, false);

            default:
                return $value;
        }
    }

/**
 * @see  FormHelper::_getModel
 */
    protected function _getModel($model)
    {
        $object = null;
        if (!$model || $model === 'Model') {
            return $object;
        }

        if (array_key_exists($model, $this->_models)) {
            return $this->_models[$model];
        }

        if (ClassRegistry::isKeySet($model)) {
            $object = ClassRegistry::getObject($model);
        } elseif (ClassRegistry::isKeySet($this->defaultModel)) {
            $defaultObject = ClassRegistry::getObject($this->defaultModel);
            if ($defaultObject && in_array($model, array_keys($defaultObject->getAssociated()), true) && isset($defaultObject->{$model})) {
                $object = $defaultObject->{$model};
            }
        } else {
            $object = ClassRegistry::init($model, true);
        }

        $this->_models[$model] = $object;
        if (!$object) {
            return null;
        }

        $this->fieldset[$model] = ['fields' => null, 'key' => $object->primaryKey, 'validates' => null];
        return $object;
    }

/**
 * @see FormHelper::_instrospectModel()
 */
    protected function _introspectModel($model = null)
    {
        if (!$model === null) {
            $model = $this->model();
        }

        $object = $this->_getModel($model);
        if (!$object) {
            return;
        }

        if (!isset($this->fieldset[$model]['fields'])) {
            $this->fieldset[$model]['fields'] = $object->schema();
        }

        return $this->fieldset[$model]['fields'];
    }

/**
 * Recebe um número de telefone como string e retorna
 * uma variante formatada.
 *
 * 33331234 -> 3333-1234
 * 6744441234 -> (67) 4444-1234
 * 11999998888 -> (11) 99999-8888
 *
 * @param  string  $number
 * @return string
 */
    public function formatPhoneNumber($number)
    {
        switch (strlen($number)) {
            case 8:
                return substr($number, 0, 4)."-".
                       substr($number, 4, 4);
            case 10:
                return "(".substr($number, 0, 2).") ".
                           substr($number, 2, 4)."-".
                           substr($number, 6, 4);
            case 11:
                return "(".substr($number, 0, 2).") ".
                           substr($number, 2, 5)."-".
                           substr($number, 7, 4);
            default:
                return $number;
        }
    }
}
